<?php

// Permitir requisições de qualquer origem
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Tratar pré-flight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

header('Content-Type: application/json; charset=utf-8');
require_once __DIR__ . '/../wp-load.php';

$conn = new mysqli(DB_HOST, DB_USER, DB_PASSWORD, DB_NAME);
if ($conn->connect_error) {
    echo json_encode(['erro' => 'Erro na conexão com o banco de dados']);
    exit;
}

$sku = isset($_GET['sku']) ? trim($_GET['sku']) : '';

if (empty($sku)) {
    echo json_encode(['erro' => 'Parâmetro "sku" obrigatório']);
    exit;
}

// Consulta o produto pelo SKU
$query = $conn->prepare("
    SELECT 
        p.ID as product_id,
        p.post_title as name,
        p.post_excerpt as short_description,
        pm_sku.meta_value as sku,
        pm_stock.meta_value as stock_status,
        pm_regular.meta_value as regular_price,
        pm_sale.meta_value as sale_price,
        pm_note.meta_value as purchase_note
    FROM wp_posts p
    LEFT JOIN wp_postmeta pm_sku     ON p.ID = pm_sku.post_id     AND pm_sku.meta_key = '_sku'
    LEFT JOIN wp_postmeta pm_stock   ON p.ID = pm_stock.post_id   AND pm_stock.meta_key = '_stock_status'
    LEFT JOIN wp_postmeta pm_regular ON p.ID = pm_regular.post_id AND pm_regular.meta_key = '_regular_price'
    LEFT JOIN wp_postmeta pm_sale    ON p.ID = pm_sale.post_id    AND pm_sale.meta_key = '_sale_price'
    LEFT JOIN wp_postmeta pm_note    ON p.ID = pm_note.post_id    AND pm_note.meta_key = '_purchase_note'
    WHERE p.post_type = 'product' AND p.post_status = 'publish' AND pm_sku.meta_value = ?
    LIMIT 1
");

$query->bind_param('s', $sku);
$query->execute();
$result = $query->get_result();

if ($result->num_rows === 0) {
    echo json_encode(['erro' => 'Produto não encontrado']);
    exit;
}

$produto = $result->fetch_assoc();

// Obter link do produto
$produto['link'] = get_permalink($produto['product_id']);

// Obter youtube_link corretamente a partir de _downloadable_files
$youtube_link = null;
$downloadable_files = get_post_meta($produto['product_id'], '_downloadable_files', true);

// var_dump($downloadable_files); // Se ainda estiver ativo para debug, remova em produção

if (!empty($downloadable_files)) {
    $files = maybe_unserialize($downloadable_files);
    if (is_array($files)) {
        foreach ($files as $item) {
            if (isset($item['file'])) {
                $youtube_link = $item['file'];
                break;
            }
        }
    }
}
$produto['youtube_link'] = $youtube_link;

// Consultar tags e converter para string
$tags = wp_get_post_terms($produto['product_id'], 'product_tag', ['fields' => 'names']);
$tags_string = implode(';', $tags);

// Consultar categorias e converter para string
$categories = wp_get_post_terms($produto['product_id'], 'product_cat', ['fields' => 'names']);
$categories_string = implode(';', $categories);

// Obter imagens do produto
$imagens = [];

// Imagem destacada
$thumb_id = get_post_thumbnail_id($produto['product_id']);
if ($thumb_id) {
    $thumb_url = wp_get_attachment_url($thumb_id);
    if ($thumb_url) {
        $imagens[] = $thumb_url;
    }
}

// Galeria de imagens
$gallery = get_post_meta($produto['product_id'], '_product_image_gallery', true);
if (!empty($gallery)) {
    $ids = explode(',', $gallery);
    foreach ($ids as $id) {
        $url = wp_get_attachment_url($id);
        if ($url) {
            $imagens[] = $url;
        }
    }
}

$fotos_string = implode(';', $imagens);


// Retornar apenas os campos solicitados
$resposta = [
    'sku'               => $produto['sku'],
    'stock_status'      => $produto['stock_status'],
    'name'              => $produto['name'],
    'short_description' => $produto['short_description'],
    'youtube_link'      => $produto['youtube_link'],
    'regular_price'     => $produto['regular_price'],
    'sale_price'        => $produto['sale_price'],
    'purchase_note'     => $produto['purchase_note'],
    'tags'              => $tags_string,
    'categories'        => $categories_string,
    'link'              => $produto['link'],
    'fotos'             => $fotos_string
];

echo json_encode([ 'data' => $resposta ], JSON_UNESCAPED_UNICODE);
