<?php

// Permitir requisições de qualquer origem (ou especifique uma origem se quiser restringir)
header('Access-Control-Allow-Origin: *');

// Permitir métodos usados por fetch ou XHR
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');

// Permitir cabeçalhos usados por fetch/XHR (Content-Type, Authorization, etc.)
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Tratar requisições OPTIONS (pré-flight) — importante para requisições com headers personalizados
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

header('Content-Type: application/json');
require_once __DIR__ . '/../wp-load.php';

$conn = new mysqli(DB_HOST, DB_USER, DB_PASSWORD, DB_NAME);
if ($conn->connect_error) {
    echo json_encode(['erro' => 'Erro na conexão com o banco de dados']);
    exit;
}

$sku_param = isset($_GET['sku']) ? trim($_GET['sku']) : '';
if (empty($sku_param)) {
    echo json_encode(['erro' => 'Parâmetro "sku" obrigatório']);
    exit;
}

$skus = array_filter(array_map('trim', explode(';', $sku_param)));
if (empty($skus)) {
    echo json_encode(['erro' => 'Nenhum SKU válido informado']);
    exit;
}

$status = $_GET['stock_status'] ?? null;
$name = $_GET['name'] ?? null;
$short_description = isset($_GET['short_description']) ? str_replace('\n', "\n", $_GET['short_description']) : null;
$youtube = $_GET['youtube_link'] ?? null;
$purchase_note = $_GET['purchase_note'] ?? null;
$regular_price = $_GET['regular_price'] ?? null;
$sale_price = $_GET['sale_price'] ?? null;
$categories = isset($_GET['categories']) ? array_filter(array_map('trim', explode(';', $_GET['categories']))) : [];

$raw_tags = $_GET['tags'] ?? '';
$tags = array_filter(array_map('trim', preg_split('/[;,]+/', $raw_tags)));


$valores_validos = ['instock', 'onbackorder', 'outofstock'];
if ($status && !in_array($status, $valores_validos)) {
    echo json_encode(['erro' => 'stock_status inválido']);
    exit;
}

$atualizados = [];
$nao_encontrados = [];

// Função para update ou insert
function update_or_insert_meta($conn, $post_id, $meta_key, $meta_value) {
    $stmt_check = $conn->prepare("SELECT meta_id FROM wp_postmeta WHERE post_id = ? AND meta_key = ?");
    $stmt_check->bind_param('is', $post_id, $meta_key);
    $stmt_check->execute();
    $stmt_check->store_result();

    if ($stmt_check->num_rows > 0) {
        // Já existe — faz UPDATE
        $stmt_update = $conn->prepare("UPDATE wp_postmeta SET meta_value = ? WHERE post_id = ? AND meta_key = ?");
        $stmt_update->bind_param('sis', $meta_value, $post_id, $meta_key);
        $stmt_update->execute();
        $stmt_update->close();
    } else {
        // Não existe — faz INSERT
        $stmt_insert = $conn->prepare("INSERT INTO wp_postmeta (post_id, meta_key, meta_value) VALUES (?, ?, ?)");
        $stmt_insert->bind_param('iss', $post_id, $meta_key, $meta_value);
        $stmt_insert->execute();
        $stmt_insert->close();
    }

    $stmt_check->close();
}

function get_or_create_term($conn, $name) {
    $stmt = $conn->prepare("SELECT term_id FROM wp_terms WHERE name = ?");
    $stmt->bind_param('s', $name);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        return $result->fetch_assoc()['term_id'];
    } else {
        $slug = custom_sanitize_title($name);
        $stmt = $conn->prepare("INSERT INTO wp_terms (name, slug) VALUES (?, ?)");
        $stmt->bind_param('ss', $name, $slug);
        $stmt->execute();
        return $stmt->insert_id;
    }
}

function get_or_create_taxonomy($conn, $term_id, $taxonomy) {
    $stmt = $conn->prepare("SELECT term_taxonomy_id FROM wp_term_taxonomy WHERE term_id = ? AND taxonomy = ?");
    $stmt->bind_param('is', $term_id, $taxonomy);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows > 0) {
        return $result->fetch_assoc()['term_taxonomy_id'];
    } else {
        $description = '';
        $parent = 0;
        $count = 0;
        $stmt = $conn->prepare("INSERT INTO wp_term_taxonomy (term_id, taxonomy, description, parent, count) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param('issii', $term_id, $taxonomy, $description, $parent, $count);
        $stmt->execute();
        return $stmt->insert_id;
    }
}

function link_post_to_taxonomy($conn, $post_id, $term_taxonomy_id) {
    $stmt = $conn->prepare("SELECT 1 FROM wp_term_relationships WHERE object_id = ? AND term_taxonomy_id = ?");
    $stmt->bind_param('ii', $post_id, $term_taxonomy_id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        $stmt = $conn->prepare("INSERT INTO wp_term_relationships (object_id, term_taxonomy_id, term_order) VALUES (?, ?, 0)");
        $stmt->bind_param('ii', $post_id, $term_taxonomy_id);
        $stmt->execute();
    }
}


function custom_sanitize_title($title) {
    $title = strtolower(trim($title));
    $title = preg_replace('/[^a-z0-9]+/', '-', $title);
    return trim($title, '-');
}

foreach ($skus as $sku) {
    $sql = "SELECT post_id FROM wp_postmeta WHERE meta_key = '_sku' AND meta_value = ? LIMIT 1";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('s', $sku);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        $nao_encontrados[] = $sku;
        continue;
    }

    $post_id = $result->fetch_assoc()['post_id'];

    // Atualizar título
    if ($name !== null) {
        $stmt = $conn->prepare("UPDATE wp_posts SET post_title = ? WHERE ID = ?");
        $stmt->bind_param('si', $name, $post_id);
        $stmt->execute();
    }

    // Atualizar descrição curta
    if ($short_description !== null) {
        $stmt = $conn->prepare("UPDATE wp_posts SET post_excerpt = ? WHERE ID = ?");
        $stmt->bind_param('si', $short_description, $post_id);
        $stmt->execute();
    }

    // Atualizar metas
    $meta_updates = [
        '_stock_status'  => $status,
        '_purchase_note' => $purchase_note,
        '_regular_price' => $regular_price,
        '_sale_price'    => $sale_price
    ];

    foreach ($meta_updates as $key => $value) {

        if ($value !== null) {
            // Verifica se já existe o meta_key para esse post_id
            $stmt_check = $conn->prepare("SELECT meta_id FROM wp_postmeta WHERE post_id = ? AND meta_key = ?");
            $stmt_check->bind_param('is', $post_id, $key);
            $stmt_check->execute();
            $stmt_check->store_result();

            if ($stmt_check->num_rows > 0) {
                // Já existe — faz UPDATE
                $stmt_update = $conn->prepare("UPDATE wp_postmeta SET meta_value = ? WHERE post_id = ? AND meta_key = ?");
                $stmt_update->bind_param('sis', $value, $post_id, $key);
                $stmt_update->execute();
                $stmt_update->close();
            } else {
                // Não existe — faz INSERT
                $stmt_insert = $conn->prepare("INSERT INTO wp_postmeta (post_id, meta_key, meta_value) VALUES (?, ?, ?)");
                $stmt_insert->bind_param('iss', $post_id, $key, $value);
                $stmt_insert->execute();
                $stmt_insert->close();
            }

            $stmt_check->close();
        }
    }


    // Atualizar youtube
    if (!empty($youtube)) {
    
        // Se já for uma URL completa, usa como está
        if (preg_match('/^https?:\/\//i', $youtube)) {
            $video_id = basename(parse_url($youtube, PHP_URL_PATH));
            if (str_contains($youtube, 'watch')) {
                parse_str(parse_url($youtube, PHP_URL_QUERY), $query_params);
                $video_id = $query_params['v'] ?? $video_id;
            }
            $youtube_url = $youtube;
        } else {
            // Caso contrário, monta a URL com base no ID
            $video_id = $youtube;
            $youtube_url = "https://www.youtube.com/watch?v=" . $youtube;
        }
    
        $download_data = [
            $video_id => [
                'name' => $video_id,
                'file' => $youtube_url
            ]
        ];
    
        // SERIALIZAR corretamente como WordPress espera
        $file_data = serialize($download_data);
    
        // Marcar como "Downloadable"
        update_or_insert_meta($conn, $post_id, '_downloadable', 'yes');
    
        // Inserir os arquivos
        update_or_insert_meta($conn, $post_id, '_downloadable_files', $file_data);
    }


    // // Atualizar categorias
    // foreach ($categories as $cat_name) {
    //     if (empty($cat_name)) continue;
    //     $term_id = get_or_create_term($conn, $cat_name);
    //     $tax_id = get_or_create_taxonomy($conn, $term_id, 'product_cat');
    //     link_post_to_taxonomy($conn, $post_id, $tax_id);
    // }

    // // Atualizar tags
    // foreach ($tags as $tag_name) {
    //     if (empty($tag_name)) continue;
    //     $term_id = get_or_create_term($conn, $tag_name);
    //     $tax_id = get_or_create_taxonomy($conn, $term_id, 'product_tag');
    //     link_post_to_taxonomy($conn, $post_id, $tax_id);
    // }
    
    // Consulta o produto pelo SKU
    $query = $conn->prepare("
        SELECT 
            p.ID as product_id,
            p.post_title as name,
            p.post_excerpt as short_description,
            pm_sku.meta_value as sku,
            pm_stock.meta_value as stock_status,
            pm_regular.meta_value as regular_price,
            pm_sale.meta_value as sale_price,
            pm_note.meta_value as purchase_note
        FROM wp_posts p
        LEFT JOIN wp_postmeta pm_sku     ON p.ID = pm_sku.post_id     AND pm_sku.meta_key = '_sku'
        LEFT JOIN wp_postmeta pm_stock   ON p.ID = pm_stock.post_id   AND pm_stock.meta_key = '_stock_status'
        LEFT JOIN wp_postmeta pm_regular ON p.ID = pm_regular.post_id AND pm_regular.meta_key = '_regular_price'
        LEFT JOIN wp_postmeta pm_sale    ON p.ID = pm_sale.post_id    AND pm_sale.meta_key = '_sale_price'
        LEFT JOIN wp_postmeta pm_note    ON p.ID = pm_note.post_id    AND pm_note.meta_key = '_purchase_note'
        WHERE p.post_type = 'product' AND p.post_status = 'publish' AND pm_sku.meta_value = ?
        LIMIT 1
    ");
    
    $query->bind_param('s', $sku);
    $query->execute();
    $result = $query->get_result();
    
    if ($result->num_rows === 0) {
        echo json_encode(['erro' => 'Produto não encontrado']);
        exit;
    }
    
    $produto = $result->fetch_assoc();
    
    // Obter link do produto
    $produto['link'] = get_permalink($produto['product_id']);
    
    // Obter youtube_link corretamente a partir de _downloadable_files
    $youtube_link = null;
    $downloadable_files = get_post_meta($produto['product_id'], '_downloadable_files', true);
    
    // var_dump($downloadable_files); // Se ainda estiver ativo para debug, remova em produção
    
    if (!empty($downloadable_files)) {
        $files = maybe_unserialize($downloadable_files);
        if (is_array($files)) {
            foreach ($files as $item) {
                if (isset($item['file'])) {
                    $youtube_link = $item['file'];
                    break;
                }
            }
        }
    }
    $produto['youtube_link'] = $youtube_link;
    
    // Consultar tags e converter para string
    $tags = wp_get_post_terms($produto['product_id'], 'product_tag', ['fields' => 'names']);
    $tags_string = implode(';', $tags);
    
    // Consultar categorias e converter para string
    $categories = wp_get_post_terms($produto['product_id'], 'product_cat', ['fields' => 'names']);
    $categories_string = implode(';', $categories);
    
    // Obter imagens do produto
    $imagens = [];
    
    // Imagem destacada
    $thumb_id = get_post_thumbnail_id($produto['product_id']);
    if ($thumb_id) {
        $thumb_url = wp_get_attachment_url($thumb_id);
        if ($thumb_url) {
            $imagens[] = $thumb_url;
        }
    }
    
    // Galeria de imagens
    $gallery = get_post_meta($produto['product_id'], '_product_image_gallery', true);
    if (!empty($gallery)) {
        $ids = explode(',', $gallery);
        foreach ($ids as $id) {
            $url = wp_get_attachment_url($id);
            if ($url) {
                $imagens[] = $url;
            }
        }
    }
    
    $fotos_string = implode(';', $imagens);
    
    
    // Retornar apenas os campos solicitados
    $atualizados[] = [
        'sku'               => $produto['sku'],
        'stock_status'      => $produto['stock_status'],
        'name'              => $produto['name'],
        'short_description' => $produto['short_description'],
        'youtube_link'      => $produto['youtube_link'],
        'regular_price'     => $produto['regular_price'],
        'sale_price'        => $produto['sale_price'],
        'purchase_note'     => $produto['purchase_note'],
        'tags'              => $tags_string,
        'categories'        => $categories_string,
        'link'              => $produto['link'],
        'fotos'             => $fotos_string
    ];
}

echo json_encode([
    'status' => 'success',
    'atualizados' => $atualizados,
    'nao_encontrados' => $nao_encontrados
]);

$conn->close();
?>
